import { Controller, Get, Post,Query, Body, Param, Put, Delete, UploadedFile, UseInterceptors, UseGuards  } from '@nestjs/common';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { CategoriesService } from './categories.service';
import { Category } from './schemas/category.schema';
import { CategoryDto } from './dto/category.dto';
import { extname } from 'path';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';

@Controller('categories')
export class CategoriesController {
  constructor(private readonly categoriesService: CategoriesService) {}

  @UseGuards(JwtAuthGuard)
  @Post()
  @UseInterceptors(
    FileInterceptor('image', {
      storage: diskStorage({
        destination: './uploads',
        filename: (req, file, cb) => {
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
          const ext = extname(file.originalname);
          cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
        },
      }),
      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png|gif)$/)) {
          return cb(new Error('Only image files are allowed!'), false);
        }
        cb(null, true);
      },
    }),
  )
  async create(@Body() categoryDto: CategoryDto, @UploadedFile() image: Express.Multer.File): Promise<Category> {
    if (image) {
      categoryDto.image = `/uploads/${image.filename}`; // Save the file path
    }
    return this.categoriesService.create(categoryDto);
  }

  @UseGuards(JwtAuthGuard)
  @Get('exists')
  async checkIfExists(@Query('name') name: string): Promise<boolean> {
    const category = await this.categoriesService.findByName(name);
    return !!category;
  }

  //@UseGuards(JwtAuthGuard)
  @Post('all')
  async findAll(): Promise<Category[]> {
    return this.categoriesService.findAll();
  }

  @Get(':id')
  async findOne(@Param('id') id: string): Promise<Category> {
    return this.categoriesService.findOne(id);
  }

  @Put(':id')
  @UseInterceptors(FileInterceptor('image'))
  async update(@Param('id') id: string, @Body() categoryDto: CategoryDto, @UploadedFile() image: Express.Multer.File): Promise<Category> {
    if (image) {
      categoryDto.image = `/uploads/${image.filename}`;
    }
    return this.categoriesService.update(id, categoryDto);
  }

  @Delete(':id')
  async delete(@Param('id') id: string): Promise<any> {
    return this.categoriesService.delete(id);
  }
}
