import { Injectable, ConflictException, InternalServerErrorException, NotFoundException  } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { Model } from 'mongoose';
import { Category } from './schemas/category.schema';
import { CategoryDto } from './dto/category.dto';

@Injectable()
export class CategoriesService {
  constructor(@InjectModel(Category.name) private categoryModel: Model<Category>) {}

  async create(categoryDto: Partial<Category>): Promise<Category> {
    try {
      const newCategory = new this.categoryModel(categoryDto);
      return await newCategory.save();
    } catch (error) {
      if (error.code === 11000) { // MongoDB duplicate key error code
        throw new ConflictException('Category name already exists.');
      }
      throw new InternalServerErrorException('Something went wrong');
    }
  }

  async findAll(): Promise<Category[]> {
    return this.categoryModel.find().exec();
  }

  async findOne(id: string): Promise<Category> {
    return this.categoryModel.findById(id).exec();
  }

  async update(id: string, categoryDto: Partial<Category>): Promise<Category> {
    try {
      return await this.categoryModel.findByIdAndUpdate(id, categoryDto, { new: true });
    } catch (error) {
      if (error.code === 11000) {
        throw new ConflictException('Category name already exists.');
      }
      throw new InternalServerErrorException('Something went wrong');
    }
  }

  async findByName(name: string): Promise<Category> {
    const category = await this.categoryModel.findOne({ name }).exec();
    if (!category) {
      throw new NotFoundException(`Category with name "${name}" not found`);
    }
    return category;
  }

  async delete(id: string): Promise<any> {
    return this.categoryModel.findByIdAndDelete(id).exec();
  }
}
