import { ConflictException, Injectable, NotFoundException, UnauthorizedException } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { Model } from 'mongoose';
import { User } from './schemas/user.schema';
import * as bcrypt from 'bcrypt';
import { CreateUserDto } from './dto/create-user.dto';
import { LoginUserDto } from './dto/login-user.dto';

@Injectable()
export class UsersService {
  constructor(@InjectModel(User.name) private userModel: Model<User>) {}

  // Create a new user
  async createUser(createUserDto: any): Promise<User> {
  	const { firstname, lastname, email, phone, usertype, userrole, password } = createUserDto;

	// Check if email already exists
	const existingUser = await this.userModel.findOne({ email });
	if (existingUser) {
	   throw new ConflictException('Email already exists');
	}

  	// Generate a random password
    //const randomPassword = Math.random().toString(36).slice(-8);

    // Hash the password
    const hashedPassword = await bcrypt.hash(password, 10);

    const newUser = new this.userModel({
      firstname,
      lastname,
      email,
      phone,
      userrole,
      usertype,
      password: hashedPassword,
    });

    await newUser.save();

    // Send email with the generated password
    //await this.sendWelcomeEmail(email, randomPassword);

    return newUser;
  }

  // Get all users
  async getAllUsers(): Promise<User[]> {
    return this.userModel.find().exec();
  }

  // Get a single user by ID
  async getUserById(id: string): Promise<User> {
    return this.userModel.findById(id).exec();
  }

  // Update a user by ID
  async updateUser(id: string, updateUserDto: any): Promise<User> {
    return this.userModel.findByIdAndUpdate(id, updateUserDto, { new: true }).exec();
  }

  // Delete a user by ID
  async deleteUser(id: string): Promise<User> {
    return this.userModel.findByIdAndDelete(id).exec();
  }

  private async sendWelcomeEmail(email: string, password: string) {
    // Implement email sending here (see Step 5)
  }

  async findByEmail(email: string): Promise<User | null> {
    return this.userModel.findOne({ email }).exec();
  }

  async comparePasswords(plainPassword: string, hashedPassword: string): Promise<boolean> {
    const bcrypt = require('bcrypt');
    return bcrypt.compare(plainPassword, hashedPassword);
  }

  async login(loginUserDto: LoginUserDto) {
    const { email, password } = loginUserDto;

    // Find user by email
    const user = await this.userModel.findOne({ email });
    if (!user) {
      throw new NotFoundException('User not found');
    }


    // Validate password
    const isPasswordValid = await bcrypt.compare(password, user.password);
    if (!isPasswordValid) {
      throw new UnauthorizedException('Invalid credentials');
    }

    // Check role and type
    if (user.userRole === 'user' && user.userType === 'user') {
      return {
        status: 'success',
        message: 'Login successful',
        data: {
          userId: user._id,
          email: user.email,
          role: 'user',
          type: 'user',
        },
      };
    }

    if (user.userRole === 'staff' || user.userRole === 'admin') {
      return {
        status: 'success',
        message: 'Login successful',
        data: {
          userId: user._id,
          email: user.email,
          role: user.userRole,
          type: user.userType,
        },
      };
    }

    // If user role/type is not handled
    throw new UnauthorizedException('Invalid user role or type');
  }
}
