import { Injectable, ConflictException, InternalServerErrorException, NotFoundException  } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { Model } from 'mongoose';
import { Category } from './schemas/category.schema';
import { CategoryDto } from './dto/category.dto';

@Injectable()
export class CategoriesService {
  constructor(@InjectModel(Category.name) private categoryModel: Model<Category>) {}

  async create(categoryDto: Partial<Category>): Promise<Category> {
    try {
      const newCategory = new this.categoryModel(categoryDto);
      return await newCategory.save();
    } catch (error) {
      if (error.code === 11000) { // MongoDB duplicate key error code
        throw new ConflictException('Category name already exists.');
      }
      throw new InternalServerErrorException('Something went wrong');
    }
  }

  async findAll(): Promise<Category[]> {
    return this.categoryModel.find().exec();
  }

  async findOne(id: string): Promise<Category> {
    return this.categoryModel.findById(id).exec();
  }

  async update(id: string, categoryDto: Partial<Category>): Promise<Category> {
    try {
      const category = await this.categoryModel.findById(id); 
      if (!category) {
        throw new Error('Product not found');
      }

      if (categoryDto.image) {
        category.image = categoryDto.image;
      } 

      Object.keys(categoryDto).forEach((key) => {
        if (key !== 'image') {
          category[key] = categoryDto[key];
        }
      });

      await category.save();

      return category;


      //return await this.categoryModel.findByIdAndUpdate(id, categoryDto, { new: true });
    } catch (error) {
      if (error.code === 11000) {
        throw new ConflictException('Category name already exists.');
      }
      throw new InternalServerErrorException('Something went wrong');
    }
  }

  async findByName(name: string): Promise<Category> {
    const category = await this.categoryModel.findOne({ name }).exec();
    if (!category) {
      throw new NotFoundException(`Category with name "${name}" not found`);
    }
    return category;
  }

  async delete(id: string): Promise<any> {
    return this.categoryModel.findByIdAndDelete(id).exec();
  }

  async findCategoryByName(name: string): Promise<Category> {
    return this.categoryModel.findOne({ name }).exec();
  }

  async findAllParent(): Promise<Category[]> {
    return this.categoryModel.find({ parent: 0 }).exec();
  }

  async findHierarchy(): Promise<Category[]> {
    const categories = await this.categoryModel.aggregate([
      {
        $graphLookup: {
          from: 'categories',
          startWith: '$_id',
          connectFromField: '_id',
          connectToField: 'parent',
          as: 'children',
        },
      },
      {
        $match: { parent: 0}, // Fetch only root categories
      },
      {
        $sort: { name: 1 }, // Optional: Sort categories alphabetically
      },
    ]);
    return categories;
  }
}
