import { Injectable, HttpException, HttpStatus, ConflictException, InternalServerErrorException, NotFoundException } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { Model,Types } from 'mongoose';
import * as crypto from 'crypto';
import axios from 'axios';
import * as dotenv from 'dotenv';

import { Product } from '../products/schemas/product.schema';
import { Cart } from '../products/schemas/cart.schema';
import { Order } from './schemas/order.schema';
import { User } from '../users/schemas/user.schema';
import { EmailService } from '../email/email.service';

dotenv.config();

@Injectable()
export class PaymentService {

  constructor(
    @InjectModel(Cart.name) private cartModel: Model<Cart>,
    @InjectModel(Order.name) private orderModel: Model<Order>,
    @InjectModel(User.name) private userModel: Model<User>,
    @InjectModel(Product.name) private productModel: Model<Product>,
    private readonly emailService: EmailService,
  ) {}
	
  private keyId = process.env.RAZORPAY_KEY_ID;
  private keySecret = process.env.RAZORPAY_SECRET;
  private razorpayBaseUrl = 'https://api.razorpay.com/v1';

  async createOrder(body,user): Promise<any> {
    try {
      const auth = Buffer.from(`${this.keyId}:${this.keySecret}`).toString('base64');
      const receipt = `receipt_${Date.now()}`;
      const response = await axios.post(
        `${this.razorpayBaseUrl}/orders`,
        {
          amount: body.amount * 100, // Amount in paise (50000 = ₹500)
          currency: 'INR',
          receipt: receipt,
        },
        {
          headers: {
            Authorization: `Basic ${auth}`,
            'Content-Type': 'application/json',
          },
        },
      );
      console.log(response.data);

     var cart = await this.cartModel.findOne({ user: new Types.ObjectId(user) }).exec();

      // Save order in MongoDB
      const order = new this.orderModel({
        amount: body.amount,
        receipt: receipt,
        orderId: response.data.id,
        address1: body.address1,
        address2: body.address2,
        address3: body.address3,
        country: body.country,
        city: body.city,
        state: body.state,
        pincode: body.pincode,
        items:cart.items,
        user:user
      });

      
      return { orderId: response.data.id, amount: body.amount, keyId: this.keyId, order: await order.save() };

      return response.data;
    } catch (error) {
      throw new HttpException(error.response?.data || 'Error creating Razorpay order', HttpStatus.BAD_REQUEST);
    }
  }

  async verifyPaymentSignature(orderId: string, paymentId: string, signature: string,user) {
    const generatedSignature = crypto
      .createHmac('sha256', this.keySecret)
      .update(`${orderId}|${paymentId}`)
      .digest('hex');

    if (generatedSignature === signature) {
      await this.orderModel.findOneAndUpdate(
        {orderId:orderId},
        {
          $set: { paymentId: paymentId, signature: signature, status:1}
        },
        { new: true, upsert: true } // `upsert: true` ensures the document is created if it doesn't exist
      );

      var products = await this.orderModel.findOne({orderId:orderId});

      await Promise.all(
        products.items.map(async (item) => {
          const product = await this.productModel.findById(item.productId);

          if (!product) return; // Skip if product not found

          let statusValue = -1; // Default status if auctionStatus is not <1
          console.log(product.auctionStatus);
          if (product.auctionStatus != '1') {
            await this.productModel.updateOne(
              { _id: product._id },
              { $set: { auctionStatus: '1', finaluser: user } }
            );
            statusValue = 1;
          }

          // ✅ Update only the matching item in the order's `items` array
          await this.orderModel.updateOne(
            { orderId: orderId, "items.productId": item.productId },
            { $set: { "items.$.status": statusValue } }
          );

          const userdata = await this.userModel.findById(user).exec();

          if(statusValue == 1) {
            var plain = "Dear " + userdata.firstname + ' ' + userdata.lastname + ", Thank you for your purchase at Travancore Heritage Auction! We are pleased to inform you that your payment has been successfully received, and your order has been confirmed. Order ID: " + orderId + ", Item Name: " + product.name + ",Amount Paid: " + products.amount + ",Payment Method: RazorPay. Our team will process your order shortly. You will receive further updates regarding dispatch and delivery. If you have any questions, feel free to reach out to us at info@travancoreheritageauction.com or call us at +91 88480 44033. Thank you for choosing Travancore Heritage Auction! Best regards, Travancore Heritage Auction Team, www.travancoreheritageauction.com";
            var content = "<p>Dear " + userdata.firstname + ' ' + userdata.lastname + ",</p><br> <p>Thank you for your purchase at Travancore Heritage Auction! We are pleased to inform you that your payment has been successfully received, and your order has been confirmed.</p><br><br> <p>Order ID: " + orderId + ",</p><br> <p>Item Name: " + product.name + ",</p><br> <p>Amount Paid: " + products.amount + "</p> <p>Payment Method: RazorPay</p><br> <p>. Our team will process your order shortly. You will receive further updates regarding dispatch and delivery. If you have any questions, feel free to reach out to us at info@travancoreheritageauction.com or call us at +91 88480 44033. </p> <br><p>Thank you for choosing Travancore Heritage Auction!.</p> <p>Best regards,<br> Travancore Heritage Auction Team,<br> www.travancoreheritageauction.com</p>";
            var subject = "Order Confirmation – Thank You for Your Purchase!";
            this.sendEmail("info@travancoreheritageauction.com",userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
            this.sendEmail(userdata.email,userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
          }

          if(statusValue != 1) {
            var plain = "Dear " + userdata.firstname + ' ' + userdata.lastname + ", Thank you for your recent purchase at Travancore Heritage Auction. Unfortunately, due to high demand, the item  " + product.name + " was sold to another buyer during the transaction processing time. We sincerely apologize for this inconvenience. We have initiated a full refund of " + products.amount + " to your original payment method. The refund should reflect in your account within [X] business days, depending on your bank's processing time. Order ID: " + orderId + ", Item Name: " + product.name + ",Amount Refunded: " + products.amount + ",Refund Status: Initiated. Our team will process your order shortly. You will receive further updates regarding dispatch and delivery. If you have any questions, feel free to reach out to us at info@travancoreheritageauction.com or call us at +91 88480 44033. Thank you for choosing Travancore Heritage Auction! Best regards, Travancore Heritage Auction Team, www.travancoreheritageauction.com";
            var content = "<p>Dear " + userdata.firstname + ' ' + userdata.lastname + ",</p><br> <p>Thank you for your recent purchase at Travancore Heritage Auction. Unfortunately, due to high demand, the item  " + product.name + " was sold to another buyer during the transaction processing time. We sincerely apologize for this inconvenience. We have initiated a full refund of " + products.amount + " to your original payment method. The refund should reflect in your account within [X] business days, depending on your bank's processing time.</p><br><br> <p>Order ID: " + orderId + ",</p><br> <p>Item Name: " + product.name + ",</p><br> <p>Amount Refunded: " + products.amount + "</p> <p>Refund Status: Initiated</p><br> <p>. Our team will process your order shortly. You will receive further updates regarding dispatch and delivery. If you have any questions, feel free to reach out to us at info@travancoreheritageauction.com or call us at +91 88480 44033. </p> <br><p>Thank you for choosing Travancore Heritage Auction!.</p> <p>Best regards,<br> Travancore Heritage Auction Team,<br> www.travancoreheritageauction.com</p>";
            var subject = "Refund Initiated for Your Purchase";
            this.sendEmail("info@travancoreheritageauction.com",userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
            this.sendEmail(userdata.email,userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
          }
        })
      );

      await this.cartModel.findOneAndUpdate({ user: new Types.ObjectId(user) }, { items: [] });
      
      return true;
    } else {
      return false;
    }
  }

  async sendEmail(to,name,plain,content,subject): Promise<any> {
      const mailjetAPIKey = 'a827249b18ef476acc5eb882fca8c1dd';
      const mailjetAPISecret = '8fb49d646c35592d0814f9ae7da771c4';

      const mailData = {
        Messages: [
          {
            From: {
              Email: "info@travancoreheritageauction.com",
              Name: "Travancore Heritage Auction"
            },
            To: [
              {
                Email: to,
                Name: name
              }
            ],
            Subject: subject,
            TextPart: plain,
            HTMLPart: content
          }
        ]
      };

      try {
        const response = await axios.post(
          'https://api.mailjet.com/v3.1/send',
          mailData,
          {
            headers: { 'Content-Type': 'application/json' },
            auth: {
              username: mailjetAPIKey,
              password: mailjetAPISecret
            }
          }
        );
        return 1;
      } catch (error) {
        return 2;
      }
  };

  async getOrder(body,user): Promise<any> {
    return await this.orderModel.findOne({orderId:body.orderId})
    .populate({
      path: 'items.productId',  // Populate product details inside each item
      model: 'Product' // Ensure this matches the actual model name in Mongoose
    })
    .populate('user');;
  }

  async getMyOrders(body,user): Promise<any> {
    return await this.orderModel.find({user:user})
    .populate({
      path: 'items.productId',  // Populate product details inside each item
      model: 'Product' // Ensure this matches the actual model name in Mongoose
    });
  }

  async getAllOrders(body,user): Promise<any> {
    return await this.orderModel.find()
    .populate({
      path: 'items.productId',  // Populate product details inside each item
      model: 'Product' // Ensure this matches the actual model name in Mongoose
    })
    .populate('user');
  }
}
