import {  Controller,  Get, Req, Post,  Query,  Body,  Param,  Put,  Delete,  UploadedFiles,  UseInterceptors,  UseGuards,  BadRequestException, UsePipes, ValidationPipe} from '@nestjs/common';
import { FilesInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { ProductService } from './product.service';
import { Product } from './schemas/product.schema';
import { ProductDto } from './dto/product.dto';
import { PaginationQueryDto } from './dto/pagination-query.dto';
import { extname } from 'path';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
const mongoose = require('mongoose');

@Controller('products')
export class ProductController {
  constructor(private readonly productService: ProductService) {}

  @UseGuards(JwtAuthGuard)
  @Post()
  @UsePipes(new ValidationPipe({ transform: true }))
  @UseInterceptors(
    FilesInterceptor('images', 5, { // Match `images[]` instead of just `images`
      storage: diskStorage({
        destination: './uploads',
        filename: (req, file, cb) => {
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
          const ext = extname(file.originalname);
          cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
        },
      }),
      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png|gif)$/)) {
          return cb(new BadRequestException('Only image files are allowed!'), false);
        }
        cb(null, true);
      },
    }),
  )
  async create(@Body() body: any, productDto: ProductDto, @UploadedFiles() files: Express.Multer.File[]): Promise<Product> {

    const transformed = new ProductDto();
    Object.assign(transformed, body); // Manually transform for debugging
    
    // Handle the uploaded files
    if (files && files.length > 0) {
      transformed.images = files.map((file) => file.filename); // Save file names
    }
    transformed.category = new mongoose.Types.ObjectId( transformed.category);
    console.log('Transformed DTO:', transformed);
    transformed.auctionStartDate= transformed.auctionStartDate ? new Date(transformed.auctionStartDate) : null;
    transformed.auctionEndDate= transformed.auctionEndDate ? new Date(transformed.auctionEndDate) : null;

    return this.productService.create(transformed);
  }

  @UseGuards(JwtAuthGuard)
  @Post('all')
  async findAll(): Promise<Product[]> {
    console.log("here");
    return this.productService.findAll();
  }

  @Post('all/pages')
  async getProducts(@Query() paginationQuery: PaginationQueryDto) {
    return this.productService.findAllPages(paginationQuery);
  }

  @Post('all-products/pages')
  async getAllProducts(@Query() @Body() paginationQuery: PaginationQueryDto) {
    return this.productService.findAllProductsPages(paginationQuery);
  }

  @Get(':id')
  async findOne(@Param('id') id: string): Promise<Product> {
    return this.productService.findOne(id);
  }

  @UseGuards(JwtAuthGuard)
  @Put(':id')
  @UsePipes(new ValidationPipe({ transform: true }))
  @UseInterceptors(
    FilesInterceptor('images', 5, { // Match `images[]` instead of just `images`
      storage: diskStorage({
        destination: './uploads',
        filename: (req, file, cb) => {
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
          const ext = extname(file.originalname);
          cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
        },
      }),
      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png|gif)$/)) {
          return cb(new BadRequestException('Only image files are allowed!'), false);
        }
        cb(null, true);
      },
    }),
  )
  async update(@Param('id') id: string, @Body() body: any, productDto: ProductDto, @UploadedFiles() files: Express.Multer.File[]): Promise<Product> {
    
    const transformed = new ProductDto();
    Object.assign(transformed, body); // Manually transform for debugging
    
    // Handle the uploaded files
    /*if (files && files.length > 0) {
      const newImages = files.map((file) => file.filename); // Save file names
    }*/
    const newImages = files && files.length > 0 ? files.map((file) => file.filename) : [];
    transformed.category = new mongoose.Types.ObjectId( transformed.category);
    console.log('Transformed DTO:', transformed);
    transformed.auctionStartDate= transformed.auctionStartDate ? new Date(transformed.auctionStartDate) : null;
    transformed.auctionEndDate= transformed.auctionEndDate ? new Date(transformed.auctionEndDate) : null;

    return this.productService.updateWithImages(id, transformed, newImages);
  }

  @UseGuards(JwtAuthGuard)
  @Delete(':id')
  async remove(@Param('id') id: string): Promise<void> {
    return this.productService.remove(id);
  }

  @Post('categories-with-count')
  async getCategoriesWithProductCount() {
    return this.productService.getCategoriesWithProductCount();
  }

  @Post('main-categories')
  async getMainCategoriesWithProductCount() {
    return this.productService.getMainCategoriesWithProductCount();
  }

  @Post('categories')
  @UseInterceptors(FilesInterceptor(''))
  async getProductsByCategory(@Body() body: any) {
    return this.productService.getProductsByCategory(body.name);
  }

  @Post('sku')
  @UseInterceptors(FilesInterceptor(''))
  async getProductBySKU(@Body() body: any) {
    return this.productService.findBySKU(body.sku);
  }

  @Post('sku-landing')
  @UseInterceptors(FilesInterceptor(''))
  async getProductBySKULanding(@Body() body: any) {
    return this.productService.getProductBySKULanding(body.sku);
  }

  @Post('lots/pages')
  @UseInterceptors(FilesInterceptor(''))
  async getProductByLots(@Query() paginationQuery: PaginationQueryDto) {
    return this.productService.findByLots(paginationQuery);
  }

  @Post('past-lots/pages')
  @UseInterceptors(FilesInterceptor(''))
  async getProductByPastLots(@Query() paginationQuery: PaginationQueryDto) {
    return this.productService.findByPastLots(paginationQuery);
  }

  @Post('store/pages')
  @UseInterceptors(FilesInterceptor(''))
  async getProductByStore(@Query() paginationQuery: PaginationQueryDto) {
    return this.productService.getProductByStore(paginationQuery);
  }

  @UseGuards(JwtAuthGuard)
  @Post('new-bid')
  @UseInterceptors(FilesInterceptor(''))
  async postNewBid(@Body() body: any,@Req() req) {
    return this.productService.newBid(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('new-reserve')
  @UseInterceptors(FilesInterceptor(''))
  async postNewReserve(@Body() body: any,@Req() req) {
    return this.productService.newReserve(body,req.user.userId);
  }

  //@UseGuards(JwtAuthGuard)
  @Post('get-bid')
  @UseInterceptors(FilesInterceptor(''))
  async getBidHistory(@Body() body: any,@Req() req) {
    return this.productService.bidHistory(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('get-secret')
  @UseInterceptors(FilesInterceptor(''))
  async getSecretHistory(@Body() body: any,@Req() req) {
    return this.productService.secretHistory(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('my-reserve')
  @UseInterceptors(FilesInterceptor(''))
  async getBidReserve(@Body() body: any,@Req() req) {
    return this.productService.bidReserve(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('addbanner')
  @UseInterceptors(FilesInterceptor(''))
  async addBanner(@Body() body: any,@Req() req) {
    return this.productService.addBanner(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('removebanner')
  @UseInterceptors(FilesInterceptor(''))
  async removeBanner(@Body() body: any,@Req() req) {
    return this.productService.removeBanner(body);
  }

  @Post('home-banners')
  @UseInterceptors(FilesInterceptor(''))
  async homeBanner(@Body() body: any,@Req() req) {
    return this.productService.homeBanners(body);
  }

  @Post('upcoming-auctions')
  @UseInterceptors(FilesInterceptor(''))
  async upcomingAuctions(@Body() body: any,@Req() req) {
    return this.productService.upcomingAuctions(body);
  }

  @Post('live-auctions')
  @UseInterceptors(FilesInterceptor(''))
  async liveAuctions(@Body() body: any,@Req() req) {
    return this.productService.liveAuctions(body);
  }

  @Post('past-auctions')
  @UseInterceptors(FilesInterceptor(''))
  async pastAuctions(@Body() body: any,@Req() req) {
    return this.productService.pastAuctions(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('switch-lot-auction')
  @UseInterceptors(FilesInterceptor(''))
  async switchLotAuction(@Body() body: any) {
    return this.productService.switchLotAuction(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('bulk-update')
  @UseInterceptors(FilesInterceptor(''))
  async bulkUpdate(@Body() body: any) {
    return this.productService.bulkUpdate(body);
  }
  

  @UseGuards(JwtAuthGuard)
  @Post('delete-product-images')
  @UseInterceptors(FilesInterceptor(''))
  async deleteImage(@Body() body: any,@Req() req) {
    return await this.productService.deleteImage(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('my-bids')
  @UseInterceptors(FilesInterceptor(''))
  async getMyBids(@Body() body: any,@Req() req) {
    return this.productService.myBidHistory(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('dashboard-counters')
  @UseInterceptors(FilesInterceptor(''))
  async getMyDashboard(@Body() body: any,@Req() req) {
    return this.productService.myDashboard(body,req.user.userId);
  }

  @Post('adjacent-products')
  @UseInterceptors(FilesInterceptor(''))
  async getAdjacentProductsBySku(@Body() body: any) {
    return this.productService.getAdjacentProductsBySku(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('all-bids')
  @UseInterceptors(FilesInterceptor(''))
  async getAllBids(@Body() body: any,@Req() req) {
    return this.productService.allBidHistory(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('request-video')
  @UseInterceptors(FilesInterceptor(''))
  async requestVideo(@Body() body: any,@Req() req) {
    return this.productService.requestVideo(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('my-video-requests')
  @UseInterceptors(FilesInterceptor(''))
  async myrequestedVideo(@Body() body: any,@Req() req) {
    return this.productService.myrequestedVideo(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('all-video-requests')
  @UseInterceptors(FilesInterceptor(''))
  async allrequestedVideo(@Body() body: any,@Req() req) {
    return this.productService.allrequestedVideo(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('current-and-next-lot')
  async getCurrentAndNextLot() {
    return await this.productService.getCurrentAndNextLot();
  }

  @Post('home-current-and-next-lot')
  async gethomeCurrentAndNextLot() {
    return await this.productService.gethomeCurrentAndNextLot();
  }

  @UseGuards(JwtAuthGuard)
  @Post('lot-bidding-result')
  @UseInterceptors(FilesInterceptor(''))
  async generateResult(@Body() body: any,@Req() req) {
    return await this.productService.generateResult(body,req.user.userId);
  }
  
  @Post('active-lot')
  async getActiveLot() {
    return await this.productService.getActiveLot();
  }
  
  @Post('next-live-lot')
  async getNextActiveLot() {
    return await this.productService.getNextLiveLot();
  }

  @Post('last-result')
  async lastResult() {
    return await this.productService.lastResult();
  }

  @UseGuards(JwtAuthGuard)
  @Post('get-cart')
  @UseInterceptors(FilesInterceptor(''))
  async getCart(@Body() body: any,@Req() req) {
    return await this.productService.getCart(req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('add-to-cart')
  @UseInterceptors(FilesInterceptor(''))
  async addToCart(@Body() body: any,@Req() req) {
    return await this.productService.addToCart(req.user.userId,body.productid,body.quantity);
  }

  @UseGuards(JwtAuthGuard)
  @Post('is-in-cart')
  @UseInterceptors(FilesInterceptor(''))
  async isProductInCart(@Body() body: any,@Req() req) {
    return await this.productService.isProductInCart(req.user.userId,body.productid);
  }

  @UseGuards(JwtAuthGuard)
  @Post('remove-from-cart')
  @UseInterceptors(FilesInterceptor(''))
  async removeFromCart(@Body() body: any,@Req() req) {
    return await this.productService.removeFromCart(req.user.userId,body.productid);
  }

  @UseGuards(JwtAuthGuard)
  @Post('clear-cart')
  @UseInterceptors(FilesInterceptor(''))
  async clearCart(@Body() body: any,@Req() req) {
    return await this.productService.clearCart(req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('change-status-manually')
  @UseInterceptors(FilesInterceptor(''))
  async changeManulStatus(@Body() body: any,@Req() req) {
    return await this.productService.changeManulStatus(body);
  }


}
