import { Injectable, ConflictException, InternalServerErrorException, NotFoundException } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { Model,Types } from 'mongoose';
import { Category } from '../categories/schemas/category.schema';
import { Product } from './schemas/product.schema';
import { Bid } from './schemas/bid.schema';
import { Cart } from './schemas/cart.schema';
import { Video } from './schemas/video.schema';
import { Result } from './schemas/result.schema';
import { Reserve } from './schemas/reserve.schema';
import { Setting } from '../settings/setting.schema';
import { User } from '../users/schemas/user.schema';
import { ProductDto } from './dto/product.dto';
import { PaginationQueryDto } from './dto/pagination-query.dto';
import { EmailService } from '../email/email.service';
const axios = require('axios');

@Injectable()
export class ProductService {
  constructor(
    @InjectModel('Category') private readonly categoryModel: Model<Category>,
    @InjectModel(Product.name) private productModel: Model<Product>,
    @InjectModel(User.name) private userModel: Model<User>,
    @InjectModel(Bid.name) private bidModel: Model<Bid>,
    @InjectModel(Cart.name) private cartModel: Model<Cart>,
    @InjectModel(Video.name) private videoModel: Model<Video>,
    @InjectModel(Result.name) private resultModel: Model<Result>,
    @InjectModel(Reserve.name) private reserveModel: Model<Reserve>,
    @InjectModel(Setting.name) private settingModel: Model<Setting>,
    private readonly emailService: EmailService,
  ) {}

  async create(productDto: Partial<Product>): Promise<Product> {
    try {
      const newProduct = new this.productModel(productDto);
      return await newProduct.save();
    } catch (error) {
      if (error.code === 11000) { // MongoDB duplicate key error code
        throw new ConflictException('Product SKU already exists.');
      }
      throw new InternalServerErrorException(error);
    }
  }

  async findAll(): Promise<Product[]> {
    return this.productModel.find().populate('category').exec();
  }

  async findOne(id: string): Promise<Product> {
    const product = await this.productModel.findById(id).populate('category').exec();
    if (!product) {
      throw new NotFoundException(`Product with id ${id} not found`);
    }
    return product;
  }

  async update(id: string, productDto: any, newImages: string[]): Promise<Product> {
    const updatedProduct = await this.productModel.findByIdAndUpdate(id, productDto, { new: true }).exec();
    if (!updatedProduct) {
      throw new NotFoundException(`Product with id ${id} not found`);
    }
    return updatedProduct;
  }

  async updateWithImages(id: string, productDto: any, newImages: string[]): Promise<Product> {
    // Use $push to append new images to the images array
    const updatedProduct = await this.productModel.findByIdAndUpdate(
      id,
      {
        $set: productDto, // Update other fields
        ...(newImages.length > 0 && { $push: { images: { $each: newImages } } }), // Add new images
      },
      { new: true } // Return the updated document
    ).exec();

    if (!updatedProduct) {
      throw new NotFoundException(`Product with id ${id} not found`);
    }

    return updatedProduct;
  }

  async remove(id: string): Promise<void> {
    const result = await this.productModel.findByIdAndDelete(id).exec();
    if (!result) {
      throw new NotFoundException(`Product with id ${id} not found`);
    }
  }

  async getCategoriesWithProductCount(): Promise<any[]> {
    return this.categoryModel.aggregate([
      {
        $lookup: {
          from: 'products', // The collection name for products
          localField: '_id', // The field in Category to match
          foreignField: 'category', // The field in Product to match
          as: 'products', // The resulting array of matched products
        },
      },
      {
        $addFields: {
          productCount: { $size: '$products' }, // Count the number of products
        },
      },
      {
        $project: {
          products: 0, // Exclude the full product list if not needed
        },
      },
    ]).exec();
  }
  
  

  /*async getMainCategoriesWithProductCount(): Promise<any[]> {
    return this.categoryModel.aggregate([
      {
        $match: {
          parent: { $eq: 0 }, // Replace 'specificField' with the actual field name
        },
      },
      {
        $lookup: {
          from: 'products', // The collection name for products
          localField: '_id', // The field in Category to match
          foreignField: 'category', // The field in Product to match
          as: 'products', // The resulting array of matched products
        },
      },
      {
        $addFields: {
          productCount: { $size: '$products' }, // Count the number of products
        },
      },
      {
        $project: {
          products: 0, // Exclude the full product list if not needed
        },
      },
    ]).exec();
  }*/
  
  async getMainCategoriesWithProductCount(): Promise<any[]> {
    return this.categoryModel.aggregate([
      {
        $match: {
          parent: 0, // Fetch only main categories (parent categories)
        },
      },
      {
        $graphLookup: {
          from: 'categories', // The collection name for categories
          startWith: '$_id', // Start with the current category
          connectFromField: '_id', // Connect using the _id of the parent category
          connectToField: 'parent', // Find categories where `parent` matches `_id`
          as: 'subcategories', // Store results in `subcategories`
        },
      },
      {
        $lookup: {
          from: 'products', // The collection name for products
          localField: '_id', // Match main category ID
          foreignField: 'category', // Match with category field in Product
          as: 'products', // Store matched products
        },
      },
      {
        $lookup: {
          from: 'products', // Again fetch products for child categories
          localField: 'subcategories._id', // Match subcategory IDs
          foreignField: 'category',
          as: 'childProducts',
        },
      },
      {
        $addFields: {
          productCount: {
            $add: [
              { $size: '$products' }, // Count main category products
              { $size: '$childProducts' }, // Count subcategory products
            ],
          },
        },
      },
      {
        $project: {
          products: 0, // Exclude full product details
          childProducts: 0,
          subcategories: 0, // Exclude subcategories if not needed
        },
      },
    ]).exec();
  }

  async getProductsByCategory(name): Promise<any[]> {
    try {
      if(!name) {
        return this.productModel.find().populate('category').exec();
      } else {
        const category = await this.categoryModel.findOne({ name: name }).exec();
        
        if (!category) {
          
          return [];
        }

        const products = await this.productModel.find({ category: category._id }).populate('category');

        return products;
      }
    } catch (error) {
      console.error('Error fetching products by category name:', error);
      throw error;
    }
  }

  async findBySKU(sku): Promise<Product> {
    //return this.productModel.findOne({ sku }).populate('category').exec();

    const product = await this.productModel
        .findOne({ sku })
        .populate('category')
        .exec();
    
      if (product && product.auctionStartDate) {
        // Subtract 5 hours and 30 minutes from the UTC time
        const utcDate = product.auctionStartDate;
        const adjustedDate = new Date(utcDate.getTime() - (5 * 60 + 30) * 60 * 1000); // Subtract 5:30
        product.auctionStartDate = adjustedDate;
      }
    
      return product;
  }
  
  async getProductBySKULanding(sku): Promise<any> {
    //return this.productModel.findOne({ sku }).populate('category').exec();

    const product = await this.productModel
        .findOne({ sku })
        .populate('category')
        .exec();
    
      if (product && product.auctionStartDate) {
        // Subtract 5 hours and 30 minutes from the UTC time
        const utcDate = product.auctionStartDate;
        const adjustedDate = new Date(utcDate.getTime() - (5 * 60 + 30) * 60 * 1000); // Subtract 5:30
        product.auctionStartDate = adjustedDate;
      }
    
      return {
        statuscode:'success',
        data: product,
        currenttime: new Date()
      };
  }

  async findByLots(paginationQuery: PaginationQueryDto): Promise<any> {
    var { page = 1, limit = 30, search, category } = paginationQuery;
    limit=32;
    const skip = (page - 1) * limit;
    
    const cats = await this.categoryModel.findOne({ name: category }).exec();
    
    if (!cats) {

        const query: any = {};
        var escapedSearch: any ='';
        if (search) {
          escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
        } 
    
        query.$and = [
          {
            $or: [
              { sku: { $regex: escapedSearch, $options: 'i' } },
              { description: { $regex: escapedSearch, $options: 'i' } },
              { shortDescription: { $regex: escapedSearch, $options: 'i' } },
              { name: { $regex: escapedSearch, $options: 'i' } },
              ...(isNaN(escapedSearch) ? [] : [{ lotNumber: { $eq: escapedSearch } }])
            ]
          },
          { lotNumber: { $ne : null } },
          { auctionStartDate: { $ne : null } },
          { auctionEndDate: { $gt: new Date() } } 
        ];
    
        //return this.productModel.find({ lotNumber: { $ne : null } }).populate('category').sort({ lotNumber: 1 }).exec();
        const products = await this.productModel.find(query).skip(skip).limit(limit).populate('category').sort({ lotNumber: 1, _id: 1 }).exec();
    
        const total = await this.productModel.countDocuments(query).exec();
    
        return {
          data: products,
          total,
          page,
          limit,
          totalPages: Math.ceil(total / limit),
        };
        
    } else {
        
        const query: any = {};
        var escapedSearch: any ='';
        if (search) {
          escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
        } 
        
        const categoryId = cats._id;
        const categories = await this.categoryModel
            .find({ $or: [{ _id: categoryId }, { parent: categoryId }] })
            .exec();
            
        const categoryIds = categories.map(category => category._id);
    
        query.$and = [
          {
            $or: [
              { sku: { $regex: escapedSearch, $options: 'i' } },
              { description: { $regex: escapedSearch, $options: 'i' } },
              { shortDescription: { $regex: escapedSearch, $options: 'i' } },
              { name: { $regex: escapedSearch, $options: 'i' } },
              ...(isNaN(escapedSearch) ? [] : [{ lotNumber: { $eq: escapedSearch } }])
            ]
          },
          { lotNumber: { $ne : null } },
          { auctionStartDate: { $ne : null } },
          { category:  { $in: categoryIds } },
          { auctionEndDate: { $gt: new Date() } } 
        ];
    
        //return this.productModel.find({ lotNumber: { $ne : null } }).populate('category').sort({ lotNumber: 1 }).exec();
        const products = await this.productModel.find(query).skip(skip).limit(limit).populate('category').sort({ lotNumber: 1, _id: 1 }).exec();
    
        const total = await this.productModel.countDocuments(query).exec();
    
        return {
          data: products,
          total,
          page,
          limit,
          totalPages: Math.ceil(total / limit),
        };
        
    }
  }

  async findByPastLots(paginationQuery: PaginationQueryDto): Promise<any> {
    var { page = 1, limit = 30, search, category, auction = '0' } = paginationQuery;
    limit=32;
    const skip = (page - 1) * limit;
    
    const cats = await this.categoryModel.findOne({ name: category }).exec();
    
    var auctionStart=new Date('2025-01-01');;
    var auctionEnd = new Date();
    
    if(auction=='1') {
        var auctionStart = new Date('2025-01-15');
        var auctionEnd = new Date('2025-02-10');
    }
    
    if(auction=='2') {
        var auctionStart = new Date('2025-06-01');
        var auctionEnd = new Date('2025-06-16');
    }
    
    if (!cats) {

        const query: any = {};
        var escapedSearch: any ='';
        if (search) {
          escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
        } 
    
        query.$and = [
          {
            $or: [
              { sku: { $regex: escapedSearch, $options: 'i' } },
              { description: { $regex: escapedSearch, $options: 'i' } },
              { shortDescription: { $regex: escapedSearch, $options: 'i' } },
              { name: { $regex: escapedSearch, $options: 'i' } },
              ...(isNaN(escapedSearch) ? [] : [{ lotNumber: { $eq: escapedSearch } }])
            ]
          },
          { lotNumber: { $ne : null } },
          { auctionStartDate: { $ne : null, $gte: auctionStart } },
          { auctionEndDate: { $lte: auctionEnd } } 
        ];
    
        //return this.productModel.find({ lotNumber: { $ne : null } }).populate('category').sort({ lotNumber: 1 }).exec();
        const products = await this.productModel.find(query).skip(skip).limit(limit).populate('category').sort({ lotNumber: 1, _id: 1 }).exec();
    
        const total = await this.productModel.countDocuments(query).exec();
    
        return {
          data: products,
          total,
          page,
          limit,
          totalPages: Math.ceil(total / limit),
        };
        
    } else {
        
        const query: any = {};
        var escapedSearch: any ='';
        if (search) {
          escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
        } 
        
        const categoryId = cats._id;
        const categories = await this.categoryModel
            .find({ $or: [{ _id: categoryId }, { parent: categoryId }] })
            .exec();
            
        const categoryIds = categories.map(category => category._id);
    
        query.$and = [
          {
            $or: [
              { sku: { $regex: escapedSearch, $options: 'i' } },
              { description: { $regex: escapedSearch, $options: 'i' } },
              { shortDescription: { $regex: escapedSearch, $options: 'i' } },
              { name: { $regex: escapedSearch, $options: 'i' } },
              ...(isNaN(escapedSearch) ? [] : [{ lotNumber: { $eq: escapedSearch } }])
            ]
          },
          { lotNumber: { $ne : null } },
          { category:  { $in: categoryIds } }, 
          { auctionStartDate: { $ne : null, $gte: auctionStart } },
          { auctionEndDate: { $lte: auctionEnd } } 
        ];
    
        //return this.productModel.find({ lotNumber: { $ne : null } }).populate('category').sort({ lotNumber: 1 }).exec();
        const products = await this.productModel.find(query).skip(skip).limit(limit).populate('category').sort({ lotNumber: 1, _id: 1 }).exec();
    
        const total = await this.productModel.countDocuments(query).exec();
    
        return {
          data: products,
          total,
          page,
          limit,
          totalPages: Math.ceil(total / limit),
        };
        
    }
  }
  
  

  async getProductByStore(paginationQuery: PaginationQueryDto): Promise<any> {
    var { page = 1, limit = 30, search, category } = paginationQuery;
    limit=32;
    const skip = (page - 1) * limit;

    const query: any = {};
    var escapedSearch: any ='';
    if (search) {
      escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
    } 

    query.$and = [
      {
        $or: [
          { sku: { $regex: escapedSearch, $options: 'i' } },
          { description: { $regex: escapedSearch, $options: 'i' } },
          { shortDescription: { $regex: escapedSearch, $options: 'i' } },
          { name: { $regex: escapedSearch, $options: 'i' } },
          ...(isNaN(escapedSearch) ? [] : [{ lotNumber: { $eq: escapedSearch } }])
        ]
      },
      { auctionStartDate:null },
    ];

    //return this.productModel.find({ lotNumber: { $ne : null } }).populate('category').sort({ lotNumber: 1 }).exec();
    const products = await this.productModel.find(query).skip(skip).limit(limit).populate('category').sort({ lotNumber: 1 }).exec();

    const total = await this.productModel.countDocuments(query).exec();

    return {
      data: products,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit),
    };
  }

  async bidHistory(body): Promise<any> {

    const product = await this.productModel.findById(body.productid).exec();
    if(product) {
      const startPrice=product.startPrice ? product.startPrice : 1;
      const maxBid = await this.bidModel
      .find({ product: body.productid })
      .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
      .limit(1) // Get the highest bid
      .exec();

      const currentMaxBidAmount = maxBid.length > 0 ? maxBid[0].bidAmount : startPrice;
      const currentMaxBidUser = maxBid.length > 0 ? maxBid[0].user : '';

      const setting = await this.settingModel.findOne({
        fromamount: { $lte: currentMaxBidAmount },
        toamount: { $gte: currentMaxBidAmount },
      });

      const bidIncrement = setting ? setting.bidincrement : 0;

      const nextBidAmount = currentMaxBidAmount + bidIncrement;


      if(maxBid.length > 0) {
        return {
          statuscode:'success',
          currentBid: currentMaxBidAmount,
          nextBid:nextBidAmount,
          currentBidUser:currentMaxBidUser,
          bid: await this.bidModel.find({ product: body.productid }).populate('user').sort({ bidAmount: -1 }).exec(),
          currenttime: new Date()
        };
      } else {
        return {
          statuscode:'success',
          currentBid: 0,
          nextBid:currentMaxBidAmount,
          currentBidUser:currentMaxBidUser,
          bid: await this.bidModel.find({ product: body.productid }).populate('user').sort({ bidAmount: -1 }).exec(),
          currenttime: new Date()
        };
      }

    } else {
      return {
        statuscode:'failure',
        message: 'Invalid Product Request',
      };
    }
  }

  async secretHistory(body): Promise<any> {

      const highestReserve = await this.reserveModel
      .find({ product: body.productid }) // Filter by productId and userId
      .populate('user')
      .sort({ reserveAmount: -1 }) // Sort by reserveAmount in descending order
      .exec();

      return {
        statuscode:'success',
        secret: highestReserve,
      };

    
  }

  async bidReserve(body,id: string): Promise<any> {

      const highestReserve = await this.reserveModel
      .findOne({ product: body.productid, user: id }) // Filter by productId and userId
      .sort({ reserveAmount: -1 }) // Sort by reserveAmount in descending order
      .exec();

      return {
        statuscode:'success',
        secret: highestReserve,
      };

    
  }

  
  async newBid(body,user): Promise<any> {
    const userdata = await this.userModel.findById(user).exec();
    const usermaxbidamount=userdata.maxBidAmount ? userdata.maxBidAmount : 0;
    const productId=body.productid;
    var amount=body.amount;
    
    if(userdata.status==0) {
      //user bidlimit not yet approved by admin
      return {
        statuscode:'failure',
        message: 'Your maximum bid limit is yet to be approved by admin. Please contact administrator',
      };
    } else {
      const product = await this.productModel.findById(body.productid).exec();
      if(product) {
          
        /* If start price not mentioned in product, start at Rs 1 */
        const startPrice=product.startPrice ? product.startPrice : 1;

        /* Check if bidding time is over */
        const currentTime = new Date();
        if (currentTime > product.auctionEndDate) {
          return {
            statuscode:'failure',
            message: 'Bidding time is over',
          };
        }

        /* check if auction status has changed and ended */
        if (product.auctionStatus!='0') {
          return {
            statuscode:'failure',
            message: 'Auction ended for this product.',
          };
        }

        /* Get the highest bid */
        const maxBid = await this.bidModel
        .find({ product: body.productid })
        .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
        .limit(1) // Get the highest bid
        .exec();
        
        /* if no amount has been passed and its the secret bid passed */
        if(amount==undefined && body.secret>0) {
            amount=body.secret;
        }
        
        /* if the highest bid user is the current user, exit , to avoid duplication or repeat*/
        if(maxBid.length>0 && maxBid[0].user.toString() === user.toString()) {
            console.log("should not add bid");
            return {
                statuscode:'failure',
                message: 'You are highest bidder now.',
            };
        }

        /* max bid is greater than the current bid or secret amount */
        if(maxBid.length>0 && maxBid[0].bidAmount>=amount) {
          return {
            statuscode:'failure',
            message: 'This bid amount already exists.',
          };
        }

        /* fix the current maximum bid, if no bids, start price will be the max bid */
        var currentMaxBidAmount = maxBid.length > 0 ? maxBid[0].bidAmount : startPrice;

        /* get next bidding increment amount  from settings */
        const setting = await this.settingModel.findOne({
          fromamount: { $lte: currentMaxBidAmount },
          toamount: { $gte: currentMaxBidAmount },
        });

        const bidIncrement = setting ? setting.bidincrement : 0;

        /* if have bids, next bid amount is current bid plus increment, else next bid is the current bid ie start price */
        if(maxBid.length > 0) {
          var nextBidAmount = currentMaxBidAmount + bidIncrement;
        } else {
          var nextBidAmount = currentMaxBidAmount;
        }

        /* get my last secret amount */
        var myreserve = await this.reserveModel
          .findOne({ product: productId, user: user })
          .sort({ reserveAmount: -1 }) // get your latest reserve
          .select('_id user reserveAmount product createdAt')
          .exec();

        /* Get highest secret  for the product */
        var highestReserve = await this.reserveModel
          .findOne({ product: productId })
          .sort({ reserveAmount: -1 })
          .select('user reserveAmount')
          .exec();
        
        /* check if my secret is highest */
        var isMyReserveHighest = true;
        if(myreserve && highestReserve) {
            isMyReserveHighest = highestReserve && highestReserve.reserveAmount === myreserve.reserveAmount && highestReserve.user.toString() === user.toString();
        } else {
            isMyReserveHighest = true;
        }
        var customreserveAmount = '';
        var sameReserveLater = '';
        
        /*check if there is any secret with my amount which came after i set the secret bid */
        if(myreserve) {
            sameReserveLater = await this.reserveModel.findOne({
                product: productId,
                reserveAmount: myreserve.reserveAmount,
                createdAt: { $gt: myreserve.createdAt },
                user: { $ne: user },
            });
        } 
        
        /* if there is other secret bid with same amount which came later, next bid amount should be my secret amount:  a change added on 10th sep, = condition added */
        if(sameReserveLater && myreserve.reserveAmount>=nextBidAmount) {
          nextBidAmount=myreserve.reserveAmount;
        }

        /* if i have the maximum bid and my bid amount is greater than all the reserve amount, stop here.*/
        if(maxBid.length>0 && maxBid[0].user==user) {

            /* check for high reserve for other users again */
            var otherhighestReserve = await this.reserveModel
            .findOne({
              product: productId,
              user: { $ne: user }, // Exclude your user
            })
            .sort({ reserveAmount: -1 })
            .select('user reserveAmount')
            .exec();
            if(otherhighestReserve) {
              if(maxBid[0].bidAmount>=otherhighestReserve.reserveAmount) {
                return {
                  statuscode:'failure',
                  message: 'You are the highest bidder now. We will let you know if you are outbidden and you may place your bid then.',
                };
              }
            } else {
                return {
                  statuscode:'failure',
                  message: 'You are the highest bidder now. We will let you know if you are outbidden and you may place your bid then.',
                };
            }
        }
        
        /* Now in to bidding procedure */
        if(usermaxbidamount<nextBidAmount) {
          return {
            statuscode:'failure',
            message: 'Your maximum bid limit is less than next bid amount',
          };
        } else {
          /* If Live auction for the product started */
          if (product.lotstart == '1') {
            const extendAuctionForSeconds = product.extendAuctionForSeconds || 20;
    
            // Find the last bid for the product
            const lastBid = await this.bidModel.find({ product: productId, createdAt: { $gt: product.productDate } })
            .sort({ createdAt: -1 }) // Sort by createdAt in descending order
            .limit(1)
            .exec();

            if (lastBid.length > 0) {
              const lastBidTime = new Date(lastBid[0].createdAt);
              const timeDifference = (currentTime.getTime() - lastBidTime.getTime()) / 1000; // Difference in seconds

              if (timeDifference > extendAuctionForSeconds) {
                if(lastBid[0].bidAmount<product.reservePrice) {
                  

                  await this.productModel.findByIdAndUpdate(
                    productId,
                    { auctionStatus: '-1' },
                  );

                  
                  return {
                    statuscode:'failure',
                    message: 'Auction has ended without being sold',
                    bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                  };
                } else {
                   await this.productModel.findByIdAndUpdate(
                      productId,
                      {
                        $set: { finaluser: lastBid[0].user },
                        $currentDate: { productDate: true }
                      }
                    );
                  
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      {
                        $set: { finalamount: lastBid[0].bidAmount },
                        $currentDate: { productDate: true }
                      }
                    );
                  
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      {
                        $set: { auctionStatus: '1' },
                        $currentDate: { productDate: true }
                      }
                    );

                  return {
                    statuscode:'failure',
                    message: `Bidding has ended for this product due to inactivity for more than ${extendAuctionForSeconds} seconds`,
                  };
                }
              } else {
                /* start bidding process, check max bid */
                var checkmaxBid = await this.bidModel
                .find({ product: body.productid })
                .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
                .limit(1) // Get the highest bid
                .exec();

                if(checkmaxBid.length>0 && checkmaxBid[0].bidAmount>=amount) {
                  return {
                    statuscode:'failure',
                    message: 'This bid amount already exists.',
                  };
                }
                
                /* if the highest bid user is the current user, exit , to avoid duplication or repeat*/
                if(checkmaxBid.length>0 && checkmaxBid[0].user.toString() === user.toString()) {
                    console.log("should not add bid");
                    return {
                        statuscode:'failure',
                        message: 'You are highest bidder now.',
                    };
                }
                
                /* Get highest secret  for the product */
                var againhighestReserve = await this.reserveModel
                  .findOne({ product: productId })
                  .sort({ reserveAmount: -1 })
                  .select('user reserveAmount')
                  .exec();
                  console.log("highest reserve" + againhighestReserve);
                 
                /* testing if adding a bid before checking reserve will sort */
                if(checkmaxBid.length>0 && checkmaxBid[0].user!=user) {
                    var result = await this.bidModel.findOneAndUpdate(
                      { product: productId, bidAmount: nextBidAmount },
                      { $setOnInsert: { user: user, timestamps: currentTime, remark: 'added from line 825' } },
                      { upsert: true, new: true }
                    );
                } 
                
                if(checkmaxBid.length==0) {
                    var result = await this.bidModel.findOneAndUpdate(
                      { product: productId, bidAmount: nextBidAmount },
                      { $setOnInsert: { user: user, timestamps: currentTime, remark: 'added from line 833' } },
                      { upsert: true, new: true }
                    );
                } 
                  
                if(againhighestReserve && againhighestReserve.reserveAmount>=amount) {
                    console.log("Reserve price > current bid("+amount+") with next bid amount as " + nextBidAmount);
                    
                    var checkmax2Bid = await this.bidModel
                    .find({ product: body.productid })
                    .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
                    .limit(1) // Get the highest bid
                    .exec();
                    
                    var current2MaxBidAmount = checkmax2Bid.length > 0 ? checkmax2Bid[0].bidAmount : startPrice;

                    const setting2 = await this.settingModel.findOne({
                      fromamount: { $lte: current2MaxBidAmount },
                      toamount: { $gte: current2MaxBidAmount },
                    });
            
                    const bid2Increment = setting ? setting.bidincrement : 0;
            
                    if(checkmax2Bid.length > 0) {
                      var next2BidAmount = current2MaxBidAmount + bid2Increment;
                    } else {
                      var next2BidAmount = current2MaxBidAmount;
                    }
                    
                    var result = await this.bidModel.findOneAndUpdate(
                      { product: productId, bidAmount: next2BidAmount },
                      { $setOnInsert: { user: againhighestReserve.user, timestamps: currentTime, remark: 'added from line 864' } },
                      { upsert: true, new: true }
                    );
                } else {
                    if(checkmaxBid[0].user==user) {
                        return {
                          statuscode:'failure',
                          message: 'You are the highest bidder now. We will let you know if you are outbidden and you may place your bid then.',
                        };
                    } else {
                        console.log("final bid from here");
                        /* commenting the code on september 10 to avoid duplication */
                        /*var result = await this.bidModel.findOneAndUpdate(
                          { product: productId, bidAmount: nextBidAmount },
                          { $setOnInsert: { user: user, timestamps: currentTime, remark: 'added from line 877' } },
                          { upsert: true, new: true }
                        );*/
                    }
                }
                
                /*var result = await this.bidModel.findOneAndUpdate(
                  { product: productId, bidAmount: nextBidAmount },
                  { $setOnInsert: { user: user, timestamps: currentTime } },
                  { upsert: true, new: true }
                );*/
            
                if (result.user.toString() != user.toString()) {
                  return {
                    statuscode:'failure',
                    message: `Bid already exists with Same Amount. Please try again.`,
                  };
                } else {
                    await this.productModel.findByIdAndUpdate(
                        productId,
                        { productDate: currentTime }
                    );
                }

                this.alertLastBidder(productId);

                
                await this.alertCurrentBidder(productId);
            

                return {
                  statuscode:'success',
                  message: 'Bid placed successfully',
                  bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                };
              }
            } else {
                const lastBidTime = new Date(product.productDate);
                const timeDifference = (currentTime.getTime() - lastBidTime.getTime()) / 1000; // Difference in seconds
        
                if (timeDifference > extendAuctionForSeconds) {
                  await this.productModel.findByIdAndUpdate(
                    productId,
                    { auctionStatus: '-1' },
                  );
                  // this.alertFloorWinner(productId);
                  return {
                    statuscode:'failure',
                    message: 'Bidding period has ended for this product due to inactivity for more than ' + extendAuctionForSeconds + ' seconds',
                  };
                } else {
                    

                    var checkmaxBidagain = await this.bidModel
                    .find({ product: body.productid })
                    .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
                    .limit(1) // Get the highest bid
                    .exec();

                    if(checkmaxBidagain.length>0 && checkmaxBidagain[0].bidAmount>=amount) {
                      return {
                        statuscode:'failure',
                        message: 'This bid amount already exists.',
                      };
                    }
                    
                    var result = await this.bidModel.findOneAndUpdate(
                      { product: productId, bidAmount: nextBidAmount },
                      { $setOnInsert: { user: user, timestamps: currentTime, remark:'this is from line 946' } },
                      { upsert: true, new: true }
                    );
                    
                    /* Get highest secret  for the product */
                    var inliveagainhighestReserve = await this.reserveModel
                      .findOne({ product: productId })
                      .sort({ reserveAmount: -1 })
                      .select('user reserveAmount')
                      .exec();
                      
                    console.log("line 957");
                    console.log(inliveagainhighestReserve);
                    console.log(nextBidAmount);
                      
                    if(inliveagainhighestReserve && inliveagainhighestReserve.reserveAmount>=nextBidAmount) {
                        
                        console.log("line 963");
                        console.log(inliveagainhighestReserve.reserveAmount);
                        console.log(nextBidAmount);
                        
                         const livesetting = await this.settingModel.findOne({
                              fromamount: { $lte: nextBidAmount },
                              toamount: { $gte: nextBidAmount },
                            });
                    
                            const livebidIncrement = livesetting ? livesetting.bidincrement : 0;
                    
                            /* if have bids, next bid amount is current bid plus increment, else next bid is the current bid ie start price */
                            var livenextBidAmount = nextBidAmount + livebidIncrement;
                            
                        if(inliveagainhighestReserve.reserveAmount>=nextBidAmount) {
                            var bidfindresult = await this.bidModel.findOneAndUpdate(
                              { product: productId, bidAmount: livenextBidAmount },
                              { $setOnInsert: { user: inliveagainhighestReserve.user, timestamps: currentTime, remark:'this is from line 962' } },
                              { upsert: true, new: true }
                            );
                        }
                    }
                
                    if (result.user.toString() != user.toString()) {
                      return {
                        statuscode:'failure',
                        message: `Bid already exists with Same Amount. Please try again.`,
                      };
                    }  else {
                        await this.productModel.findByIdAndUpdate(
                            productId,
                            { productDate: currentTime }
                        );
                    }
        
                    this.alertLastBidder(productId);
        
                    await this.alertCurrentBidder(productId);

                    return {
                        statuscode:'success',
                        message: 'Bid placed successfully',
                        /*bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),*/
                    };
                }
              
            }
          } else {
            
            /* again fetch max bid for the product */
            var checkmaxBidagainsecond = await this.bidModel
            .find({ product: body.productid })
            .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
            .limit(1) // Get the highest bid
            .exec();


            if(checkmaxBidagainsecond.length>0 && checkmaxBidagainsecond[0].bidAmount>=amount) {
              return {
                statuscode:'failure',
                message: 'This bid amount already exists.',
              };
            }
            
            /* if the highest bid user is the current user, exit , to avoid duplication or repeat*/
            if(checkmaxBidagainsecond.length>0 && checkmaxBidagainsecond[0].user.toString() === user.toString()) {
                console.log("again should not add bid");
                return {
                    statuscode:'failure',
                    message: 'You are highest bidder now.',
                };
            }
            
            /* Get highest secret for the product */
            var againhighestReserve = await this.reserveModel
              .findOne({ product: productId })
              .sort({ reserveAmount: -1 })
              .select('user reserveAmount')
              .exec();
            
            /* if reserve amount is greater than current bid amount, add in reserve users name, else current bidders */
            if(againhighestReserve && againhighestReserve.reserveAmount>=amount) {
                
                /* get next bidding increment amount  from settings */
                const finalsetting = await this.settingModel.findOne({
                  fromamount: { $lte: nextBidAmount },
                  toamount: { $gte: nextBidAmount },
                });
        
                const finalbidIncrement = finalsetting ? finalsetting.bidincrement : 0;
        
                /* if have bids, next bid amount is current bid plus increment, else next bid is the current bid ie start price 
                if(checkmaxBidagainsecond.length > 0) {
                  var nextBidAmount = nextBidAmount + finalbidIncrement;
                } else {
                  var nextBidAmount = nextBidAmount;
                }*/
                var bidfindresult = await this.bidModel.findOneAndUpdate(
                  { product: productId, bidAmount: nextBidAmount },
                  { $setOnInsert: { user: againhighestReserve.user, timestamps: currentTime, remark:'this is from line 1026' } },
                  { upsert: true, new: true }
                );
            } else {
                var bidfindresult = await this.bidModel.findOneAndUpdate(
                  { product: productId, bidAmount: nextBidAmount },
                  { $setOnInsert: { user: user, timestamps: currentTime, remark:'this is from line 1007' } },
                  { upsert: true, new: true }
                );
            }
            
            
            /* if last bid was inserted or not */
            if (bidfindresult.user.toString() != user.toString()) {
              return {
                statuscode:'failure',
                message: `Bid already exists with Same Amount. Please try again.`,
              };
            } else {
                await this.productModel.findByIdAndUpdate(
                    productId,
                    { productDate: currentTime }
                );
            }

            this.alertLastBidder(productId);
            
            /* get top reserve again */
            const result = await this.reserveModel
            .findOne({ product: productId })
            .sort({ reserveAmount: -1 })
            .select('_id user reserveAmount product')
            .exec();

            if(result && result.reserveAmount>nextBidAmount) {
                
              /* Get next reserve after top reserve */
              var nextReserve = await this.reserveModel
              .find({ product:productId }) // Add the where condition
              .sort({ reserveAmount: -1 }) // Sort by created_at descending
              .skip(1) // Skip the first document
              .limit(1) // Limit to one document
              .exec();

              /* if next reserve amount is greater than next bid amount, set the next reserve amount amount as next bid amount */
              if(nextReserve.length>0 && nextReserve[0].reserveAmount>nextBidAmount) {
                nextBidAmount=nextReserve[0].reserveAmount;
                
                var newsetting = await this.settingModel.findOne({
                    fromamount: { $lte: nextBidAmount },
                    toamount: { $gte: nextBidAmount },
                  });
    
                  var newbidIncrement = newsetting ? newsetting.bidincrement : 0;
    
                  var newnextBidAmount = nextBidAmount + newbidIncrement;
                  if(result.reserveAmount>=newnextBidAmount) {
                    const morenewBid = new this.bidModel({
                      user: result.user,
                      product: result.product,
                      bidAmount: newnextBidAmount,
                      timestamps: currentTime,
                    });
    
                    await morenewBid.save();
                    this.alertLastBidder(result.product);
                  }
              }

              
            }
            
            await this.alertCurrentBidder(productId);
            
            return {
              statuscode:'success',
              message: 'Bid placed successfully',
              /*bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),*/
            };
          }
          
        }


        

      } else {
        return {
          statuscode:'failure',
          message: 'Invalid Product Request',
        };
      }
      //return await this.bidModel.find({ product: body.productid }).populate('user').exec();;
    }

  }

  async alertLastBidder(productId): Promise<any> {
    
    const lastbid = await this.bidModel
      .find({ product:productId })
      .sort({ createdAt: -1 }) // Sort by created_at descending
      .skip(1) // Skip the first document
      .limit(1) // Limit to one document
      .exec(); // Execute the query
    
    if(lastbid.length>0) {
      const userdata = await this.userModel.findById(lastbid[0].user).exec();
      const productdata = await this.productModel.findById(productId).exec();
      
      
        const highestBid = await this.bidModel
          .find({ product:productId })
          .sort({ createdAt: -1 }) // Sort by created_at descending
          .limit(1) // Limit to one document
          .exec();

        
      var plain = "Dear " + userdata.firstname + ' ' + userdata.lastname + ", Your Bid was Outbid - Place a New Bid Now! You're still in the Auction! Don't Miss out on the opportunity to win this item. Place a new bid now to stay ahead!";
      var content = "<p>Dear " + userdata.firstname + ' ' + userdata.lastname + ",</p> <br><p>You’ve been outbid on Lot #"+productdata.lotNumber+" – "+productdata.name+".</p><br><p>&#9888;&#65039; <strong>Highest Bid Now:</strong> ₹"+highestBid[0].bidAmount+"</p><p>&#9989; <Strong>Your Bid:</strong> "+lastbid[0].bidAmount+"</p><br><p>Don’t miss your chance! Place a higher bid before the auction ends:</p><br><p>&#128073; <a href='https://travancoreheritageauction.com/product-view/"+productdata.sku+"'>Place a New Bid Now</a></p><br><p><Strong>Good luck!</strong></p><p>Travancore Heritage Auction Team</p>";
      //var subject = "Your Bid was Outbid - Travancore Heritage Auction";
      var subject = "You’ve Been Outbid on Lot #"+productdata.lotNumber+" – Place a New Bid Now!"
      this.sendEmail(userdata.email,userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
    }
  }

  async alertCurrentBidder(productId): Promise<any> {
 
    const lastbid = await this.bidModel
      .find({ product:productId })
      .sort({ createdAt: -1 }) // Sort by created_at descending
      .limit(1) // Limit to one document
      .exec(); // Execute the query
    if(lastbid.length>0) {

      const userdata = await this.userModel.findById(lastbid[0].user).exec();
      const productdata = await this.productModel.findById(productId).exec();
      var plain = "Dear Admin, Mr " + userdata.firstname + ' ' + userdata.lastname + ", has successfully placed a bid for the following item. Cust ID: THA" + userdata.regno + ", Lot Number: " + productdata.lotNumber + ", Bid Value: " + lastbid[0].bidAmount;
      var content = "<p>Dear Admin, Mr " + userdata.firstname + ' ' + userdata.lastname + ", has successfully placed a bid for the following item.</p><br><p>Cust ID: THA" + userdata.regno + "</p><br><p>Lot Number: " + productdata.lotNumber + "</p><br><p>Bid Value: " + lastbid[0].bidAmount + "</p>";
      var subject = "New Bid From Customer - Travancore Heritage Auction";
      this.sendEmail('travancoreheritageauction@gmail.com',userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
      //this.sendEmail('indrajithkk1997@gmail.com',userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
      
        const wresponse = await axios.post(
          'https://graph.facebook.com/v23.0/709680682221149/messages',
          {
            messaging_product: 'whatsapp',
            to: '918848044033',
            type: 'template',
            template: {
              name: 'new_bid_placed_admin_notification_new',
              language: {
                code: 'en',
              },
              components: [
                {
                  type: 'body',
                  parameters: [
                    {
                      type: 'text',
                      parameter_name:'name',
                      text: userdata.firstname + ' ' + userdata.lastname, 
                    },
                    {
                      type: 'text',
                      parameter_name:'id',
                      text: userdata.regno, 
                    },
                    {
                      type: 'text',
                      parameter_name:'number',
                      text: userdata.phone, 
                    },
                    {
                      type: 'text',
                      parameter_name:'bidamount',
                      text: lastbid[0].bidAmount, 
                    },
                    {
                      type: 'text',
                      parameter_name:'lotnumber',
                      text: productdata.lotNumber, 
                    },
                    {
                      type: 'text',
                      parameter_name:'itemname',
                      text: productdata.name, 
                    },
                  ],
                },
              ],
            },
          },
          {
            headers: {
              'Content-Type': 'application/json',
              Authorization: 'Bearer EAAFVXw8ns9ABO7hGWIU57K1PZBOlvDtylqUt8AfVF5AXt5i8aTZAa2qeqcVpE9eorejrLggCrIgBau77usFS7CZAZAtXxyl2RK9Mh4Qhw4ZBPUq6s4ArkvM8VfPVnNm58ua6jXhOTzhfqCcrSzPzpjpSniq36tGKICBxqEjJfqSfK68Vqbb0stCuORwOd6kGEHGCPFHOIQlsiC9R2FDXdGHxpZBZBXTZCEcyrU4nw1wLugnjS5T2OtvIhjYZB00EiZAV6I2sjsEuvC5QZDZD',
            },
          },
    
    
        );
    }
  }

  async sendEmail(to,name,plain,content,subject): Promise<any> {
      const mailjetAPIKey = 'a827249b18ef476acc5eb882fca8c1dd';
      const mailjetAPISecret = '8fb49d646c35592d0814f9ae7da771c4';

      const mailData = {
        Messages: [
          {
            From: {
              Email: "info@travancoreheritageauction.com",
              Name: "Travancore Heritage Auction"
            },
            To: [
              {
                Email: to,
                Name: name
              }
            ],
            Subject: subject,
            TextPart: plain,
            HTMLPart: content
          }
        ]
      };

      try {
        const response = await axios.post(
          'https://api.mailjet.com/v3.1/send',
          mailData,
          {
            headers: { 'Content-Type': 'application/json' },
            auth: {
              username: mailjetAPIKey,
              password: mailjetAPISecret
            }
          }
        );
        return 1;
      } catch (error) {
        return 2;
      }
  };

  /*async newReserve(body,user): Promise<any> {
    const userdata = await this.userModel.findById(user).exec();
    const usermaxbidamount=userdata.maxBidAmount ? userdata.maxBidAmount : 0;
    const productId=body.productid;
    const secretamount=body.secret;
    if(userdata.status==0) {
      //user bidlimit not yet approved by admin
      return {
        statuscode:'failure',
        message: 'Your maximum bid limit is yet to be approved by admin. Please contact administrator',
      };
    } else {
      const product = await this.productModel.findById(body.productid).exec();
      if(product) {
        const startPrice=product.startPrice ? product.startPrice : 1;

        const currentTime = new Date();
        if (currentTime > product.auctionEndDate) {
          return {
            statuscode:'failure',
            message: 'Bidding time is over',
          };
        }
        
        if(secretamount<startPrice) {
            return {
            statuscode:'failure',
            message: 'Secret Amount should be atleast the Minimum Bid of the Product',
          };
        }

        const maxBid = await this.bidModel
        .find({ product: body.productid })
        .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
        .limit(1) // Get the highest bid
        .exec();

        const currentMaxBidAmount = maxBid.length > 0 ? maxBid[0].bidAmount : startPrice;


        const setting = await this.settingModel.findOne({
          fromamount: { $lte: currentMaxBidAmount },
          toamount: { $gte: currentMaxBidAmount },
        });

        const bidIncrement = setting ? setting.bidincrement : 0;

        //const nextBidAmount = currentMaxBidAmount + bidIncrement;
        if(maxBid.length > 0) {
          var nextBidAmount = currentMaxBidAmount + bidIncrement;
        } else {
          var nextBidAmount = currentMaxBidAmount;
        }

        if(usermaxbidamount<=nextBidAmount) {
          return {
            statuscode:'failure',
            message: 'Your maximum bid limit is less than next bid amount based on current secret bid. So your secret bid could not be placed.',
          };
        } else {
          if (product.lotstart == '1' && product.extendAuctionOnBid) {
            const extendAuctionForSeconds = product.extendAuctionForSeconds || 20;

            // Find the last bid for the product
            const lastBid = await this.bidModel.find({ product: productId })
            .sort({ createdAt: -1 }) // Sort by createdAt in descending order
            .limit(1)
            .exec();

            if (lastBid.length > 0) {
              const lastBidTime = new Date(lastBid[0].createdAt);
              const timeDifference = (currentTime.getTime() - lastBidTime.getTime()) / 1000; // Difference in seconds

              if (timeDifference > extendAuctionForSeconds) {
                if(lastBid[0].bidAmount<product.reservePrice && product.reservePrice>product.startPrice) {
                  const newReserve = new this.reserveModel({
                    user: user,
                    product: productId,
                    reserveAmount: secretamount,
                    timestamps: currentTime,
                  });

                  await newReserve.save();

                  var maxReserveDetails = await this.reserveModel
                  .find({ product: body.productid })
                  .sort({ reserveAmount: -1 }) // Sort by bidAmount in descending order
                  .limit(1) // Get the highest bid
                  .exec();

                  if(maxReserveDetails.length>0) {
                    if(secretamount<maxReserveDetails[0].reserveAmount) {
                      var reservebidresult = await this.newBid(body,maxReserveDetails[0].user);
                    } else {
                        if(secretamount>=nextBidAmount) {
                            var reservebidresult = await this.newBid(body,user);
                        }
                    }
                  } else {
                    if(secretamount>=nextBidAmount) {
                        var reservebidresult = await this.newBid(body,user);
                    }
                  }              

                  const product = await this.productModel.findById(productId);

                  const updatedProduct = await this.productModel.findOneAndUpdate(
                    { 
                      _id: productId, 
                      $or: [
                        { userReservePrice: { $lt: secretamount } }, // Field exists and is less than secretamount
                        { userReservePrice: { $exists: false } }    // Field does not exist
                      ]
                    },
                    { 
                      userReservePrice: secretamount 
                    },
                    { 
                      new: true, // Return the updated document
                      fields: { userReservePrice: 1 }, // Return only the userReservePrice field
                    }
                  );


                  if (updatedProduct) {
                    return {
                      statuscode:'success',
                      message: 'Secret Maximum Bid placed successfully.',
                      bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                      secret:updatedProduct.userReservePrice,
                    };
                  } else {
                    return {
                      statuscode:'success',
                      message: 'Secret Maximum Bid could not be placed as someone has placed higher secret bid.',
                      bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                      secret:0,
                    };
                  }
                } else {
                  await this.productModel.findByIdAndUpdate(
                    productId,
                    { finaluser: lastBid[0].user },
                  );
                  await this.productModel.findByIdAndUpdate(
                    productId,
                    { finalamount: lastBid[0].bidAmount },
                  );

                  return {
                    statuscode:'failure',
                    message: `Bidding period has ended for this product due to inactivity for more than ${extendAuctionForSeconds} seconds`,
                  };
                }
              } else {
                const newReserve = new this.reserveModel({
                  user: user,
                  product: productId,
                  reserveAmount: secretamount,
                  timestamps: currentTime,
                });

                await newReserve.save();

                  var maxReserveDetails = await this.reserveModel
                  .find({ product: body.productid })
                  .sort({ reserveAmount: -1 }) // Sort by bidAmount in descending order
                  .limit(1) // Get the highest bid
                  .exec();

                  if(maxReserveDetails.length>0) {
                    if(secretamount<maxReserveDetails[0].reserveAmount) {
                      var reservebidresult = await this.newBid(body,maxReserveDetails[0].user);
                    } else {
                        if(secretamount>=nextBidAmount) {
                            var reservebidresult = await this.newBid(body,user);
                        }
                    }
                  } else {
                        if(secretamount>=nextBidAmount) {
                            var reservebidresult = await this.newBid(body,user);
                        }
                  }         

                const updatedProduct = await this.productModel.findOneAndUpdate(
                  { 
                    _id: productId, 
                    $or: [
                      { userReservePrice: { $lt: secretamount } }, // Field exists and is less than secretamount
                      { userReservePrice: { $exists: false } }    // Field does not exist
                    ]
                  },
                  { 
                    userReservePrice: secretamount 
                  },
                  { 
                    new: true, // Return the updated document
                    fields: { userReservePrice: 1 }, // Return only the userReservePrice field
                  }
                );

                if (updatedProduct) {
                  return {
                    statuscode:'success',
                    message: 'Secret Maximum Bid placed successfully.',
                    bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                    secret:updatedProduct.userReservePrice,
                  };
                } else {
                  return {
                    statuscode:'success',
                    message: 'Secret Maximum Bid could not be placed as someone has placed higher secret bid.',
                    bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                    secret:0,
                  };
                }
              }
            } else {
              const newReserve = new this.reserveModel({
                user: user,
                product: productId,
                reserveAmount: secretamount,
                timestamps: currentTime,
              });

              await newReserve.save();

              var maxReserveDetails = await this.reserveModel
              .find({ product: body.productid })
              .sort({ reserveAmount: -1 }) // Sort by bidAmount in descending order
              .limit(1) // Get the highest bid
              .exec();

              if(maxReserveDetails.length>0) {
                if(secretamount<maxReserveDetails[0].reserveAmount) {
                  var reservebidresult = await this.newBid(body,maxReserveDetails[0].user);
                } else {
                        if(secretamount>=nextBidAmount) {
                            var reservebidresult = await this.newBid(body,user);
                        }
                }
              } else {
                        if(secretamount>=nextBidAmount) {
                            var reservebidresult = await this.newBid(body,user);
                        }
              }    

              const updatedProduct = await this.productModel.findOneAndUpdate(
                { 
                  _id: productId, 
                  $or: [
                    { userReservePrice: { $lt: secretamount } }, // Field exists and is less than secretamount
                    { userReservePrice: { $exists: false } }    // Field does not exist
                  ]
                },
                { 
                  userReservePrice: secretamount 
                },
                { 
                  new: true, // Return the updated document
                  fields: { userReservePrice: 1 }, // Return only the userReservePrice field
                }
              );

              if (updatedProduct) {
                return {
                  statuscode:'success',
                  message: 'Secret Maximum Bid placed successfully.',
                  bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                  secret:updatedProduct.userReservePrice,
                };
              } else {
                return {
                  statuscode:'success',
                  message: 'Secret Maximum Bid could not be placed as someone has placed higher secret bid.',
                  bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                  secret:0,
                };
              }
            }
          } else {
            const newReserve = new this.reserveModel({
              user: user,
              product: productId,
              reserveAmount: secretamount,
              timestamps: currentTime,
            });

            var maxReserveDetails = await this.reserveModel
            .find({ product: body.productid })
            .sort({ reserveAmount: -1 }) // Sort by bidAmount in descending order
            .limit(1) // Get the highest bid
            .exec();

            await newReserve.save();

            if(maxReserveDetails.length>0) {
              if(secretamount<maxReserveDetails[0].reserveAmount) {
                var reservebidresult = await this.newBid(body,maxReserveDetails[0].user);
              } else {
                  
                        if(secretamount>=nextBidAmount) {
                            var reservebidresult = await this.newBid(body,user);
                        }
              }
            } else {
              if(secretamount>=nextBidAmount) {
                    var reservebidresult = await this.newBid(body,user);
                }
            }   

            const updatedProduct = await this.productModel.findOneAndUpdate(
              { 
                _id: productId, 
                $or: [
                  { userReservePrice: { $lt: secretamount } }, // Field exists and is less than secretamount
                  { userReservePrice: { $exists: false } }    // Field does not exist
                ]
              },
              { 
                userReservePrice: secretamount 
              },
              { 
                new: true, // Return the updated document
                fields: { userReservePrice: 1 }, // Return only the userReservePrice field
              }
            );

            if (updatedProduct) {
              return {
                statuscode:'success',
                message: 'Secret Maximum Bid placed successfully.',
                bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                secret:updatedProduct.userReservePrice,
              };
            } else {
              return {
                statuscode:'success',
                message: 'Secret Maximum Bid could not be placed as someone has placed higher secret bid.',
                bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                secret:0,
              };
            }
          }
          
        }

      } else {
        return {
          statuscode:'failure',
          message: 'Invalid Product Request',
        };
      }
      //return await this.bidModel.find({ product: body.productid }).populate('user').exec();;
    }
  }*/
  
  async newReserve(body, user): Promise<any> {
    const userdata = await this.userModel.findById(user).exec();
    const usermaxbidamount = userdata.maxBidAmount ? userdata.maxBidAmount : 0;
    const productId = body.productid;
    const secretamount = body.secret;

    if (userdata.status == 0) {
      return {
        statuscode: 'failure',
        message:
          'Your maximum bid limit is yet to be approved by admin. Please contact administrator',
      };
    } else {
      const product = await this.productModel.findById(body.productid).exec();
      if (product) {
        const startPrice = product.startPrice ? product.startPrice : 1;
        const currentTime = new Date();

        if (currentTime > product.auctionEndDate) {
          return {
            statuscode: 'failure',
            message: 'Bidding time is over',
          };
        }

        if (secretamount < startPrice) {
          return {
            statuscode: 'failure',
            message:
              'Secret Amount should be atleast the Minimum Bid of the Product',
          };
        }

        const maxBid = await this.bidModel
          .find({ product: body.productid })
          .sort({ bidAmount: -1 })
          .limit(1)
          .exec();

        const currentMaxBidAmount =
          maxBid.length > 0 ? maxBid[0].bidAmount : startPrice;

        const setting = await this.settingModel.findOne({
          fromamount: { $lte: currentMaxBidAmount },
          toamount: { $gte: currentMaxBidAmount },
        });

        const bidIncrement = setting ? setting.bidincrement : 0;
        var nextBidAmount =
          maxBid.length > 0
            ? currentMaxBidAmount + bidIncrement
            : currentMaxBidAmount;

        if (secretamount < nextBidAmount) {
          return {
            statuscode: 'failure',
            message:
              'Secret Amount should be greater than the next Bid amount of the Product',
          };
        }

        if (usermaxbidamount <= nextBidAmount) {
          return {
            statuscode: 'failure',
            message:
              'Your maximum bid limit is less than next bid amount based on current secret bid. So your secret bid could not be placed.',
          };
        } else {
          /*const newReserve = new this.reserveModel({
            user: user,
            product: productId,
            reserveAmount: secretamount,
            timestamps: currentTime,
          });
          await newReserve.save();*/

          let placeNewBid = false;
          let bidUser = null;

          var maxReserveDetails = await this.reserveModel
            .find({ product: body.productid })
            .sort({ reserveAmount: -1 })
            .limit(1)
            .exec();
            
        const newReserve = new this.reserveModel({
            user: user,
            product: productId,
            reserveAmount: secretamount,
            timestamps: currentTime,
          });
          await newReserve.save();

          if (maxReserveDetails.length > 0) {
            if (secretamount <= maxReserveDetails[0].reserveAmount && user != maxReserveDetails[0].user) {
              placeNewBid = true;
              bidUser = maxReserveDetails[0].user;
            } else {
              if (secretamount >= nextBidAmount) {
                placeNewBid = true;
                bidUser = user;
              }
            }
          } else {
            if (secretamount >= nextBidAmount) {
              placeNewBid = true;
              bidUser = user;
            }
          }

          if (placeNewBid) {
            var reservebidresult = await this.newBid(body, bidUser);
          }

          const updatedProduct = await this.productModel.findOneAndUpdate(
            {
              _id: productId,
              $or: [
                { userReservePrice: { $lt: secretamount } },
                { userReservePrice: { $exists: false } },
              ],
            },
            {
              userReservePrice: secretamount,
            },
            {
              new: true,
              fields: { userReservePrice: 1 },
            },
          );
          console.log("1");
          console.log(updatedProduct);

          if (updatedProduct) {
              console.log("reurning 2");
            return {
              statuscode: 'success',
              message: 'Secret Maximum Bid placed successfully.',
              bid: await this.bidModel
                .find({ product: body.productid })
                .populate('user')
                .exec(),
              secret: updatedProduct.userReservePrice,
            };
          } else {
            var highestReserve = await this.reserveModel
              .findOne({ product: body.productid })
              .sort({ reserveAmount: -1 })
              .select('user reserveAmount')
              .exec();
              if(user != highestReserve.user) {
                var reservebidresult = await this.newBid(body, highestReserve.user);
                return {
                  statuscode: 'success',
                  message:
                    'Secret Maximum Bid could not be placed as someone has placed higher secret bid.',
                  bid: await this.bidModel
                    .find({ product: body.productid })
                    .populate('user')
                    .exec(),
                  secret: 0,
                  hig: highestReserve,
                };
              } else {
                return {
                  statuscode: 'success',
                  message:'Secret bid Placed.',
                  bid: await this.bidModel
                    .find({ product: body.productid })
                    .populate('user')
                    .exec(),
                  secret: 0,
                  hig: highestReserve,
                };  
              }
          }
        }
      } else {
        return {
          statuscode: 'failure',
          message: 'Invalid Product Request',
        };
      }
    }
  }


  async addBanner(body): Promise<any> {
    await this.productModel.findByIdAndUpdate(
      body.id,
      { isbanner: 'yes' },
    );
    return {
      statuscode:'success',
      message: 'Product will be available in Home Banner',
    };
  }

  async removeBanner(body): Promise<any> {
    await this.productModel.findByIdAndUpdate(
      body.id,
      { isbanner: 'no' },
    );
    return {
      statuscode:'success',
      message: 'Product will be removed from Home Banner',
    };
  }

  async homeBanners(body): Promise<any> {
    return this.productModel.find({isbanner:'yes'}).populate('category').exec();
  }

  async upcomingAuctions(body): Promise<any> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    return await this.productModel
    .find({
      $and: [
        { auctionStartDate: { $gt: today } },
        { lotNumber: { $ne: null } }
      ]
    })
    .populate('category')
    .limit(6)
    .exec();
  }

  async liveAuctions(body): Promise<any> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    
    const tomorrow = new Date(today);
    tomorrow.setDate(tomorrow.getDate() + 1);
    
    return this.productModel
      .find({
        $and: [
          { auctionStartDate: { $lt: tomorrow } },
          { auctionEndDate: { $gte: today } },
          { lotNumber: { $ne: null } }
        ]
      })
      .populate('category')
      .limit(6)
      .exec();
  }

  async pastAuctions(body): Promise<any> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    return this.productModel
    .find({
      $and: [
        { auctionEndDate: { $lt: today } },
        { lotNumber: { $ne: null } }
      ]
    })
    .populate('category')
    .limit(6)
    .exec();
  }

  async switchLotAuction(body): Promise<any> {

    const product = await this.productModel.findById(body.id).exec();

    if (!product) {
      throw new Error('Product not found');
    }
    const newLotStart = body.status == '0' ? 1 : 0;

    const updatedProduct = await this.productModel.findByIdAndUpdate(
      body.id,
      {
        $set: { lotstart: product.lotstart !== undefined ? newLotStart : 0 },
        $currentDate: {
          productDate: true // This sets `productdate` to the current date and time
        }
      },
      { new: true, upsert: true } // `upsert: true` ensures the document is created if it doesn't exist
    );

    return {
      statuscode:'success',
      message: 'LOT Status Updated.',
      data:updatedProduct
    };
  }

  async bulkUpdate(body): Promise<any> {
    const filter = {  }; 
    const update = { $set: { auctionStartDate: body.auctionStartDate } }; 

    const result = await this.productModel.updateMany(filter, update);
    return result;
  }

  async findAllPages(paginationQuery: PaginationQueryDto) {
    const { page = 1, limit = 10, search } = paginationQuery;
    const skip = (page - 1) * limit;

    
    const query: any = {};
    if (search) {
      const escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
      //query.sku = { $regex: escapedSearch, $options: 'i' },

      query.$or = [
        { sku: { $regex: escapedSearch, $options: 'i' } },
        { $expr: { $regexMatch: { input: { $toString: "$lotNumber" }, regex: escapedSearch, options: 'i' } } },
        { description: { $regex: search, $options: 'i' } },
        { shortDescription: { $regex: search, $options: 'i' } },
        { name: { $regex: search, $options: 'i' } }
      ];
    }

    const products = await this.productModel
      .find(query)
      .skip(skip)
      .limit(limit)
      .exec();

    const total = await this.productModel.countDocuments().exec();

    return {
      data: products,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit),
    };
  }

  async findAllProductsPages(paginationQuery: PaginationQueryDto) {
    var { page = 1, limit = 27, search, category, fromprice = 0, toprice = 99999999, itemstatus = '', material ='' } = paginationQuery;
    limit=27;
    const skip = (page - 1) * limit;
    if(fromprice==null) {
        fromprice=0;
    }
    if(toprice==null) {
        toprice=9999999;
    }
    if(!fromprice) {
        fromprice=0;
    }
    if(!toprice) {
        toprice=9999999;
    }

    const cats = await this.categoryModel.findOne({ name: category }).exec();
    
    if (!cats) {

      const query: any = {};
      var escapedSearch: any ='';
      if (search) {
        escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
        //query.sku = { $regex: escapedSearch, $options: 'i' },
      } 

        query.$and = [
          {
            $or: [
              { sku: { $regex: escapedSearch, $options: 'i' } },
              { description: { $regex: escapedSearch, $options: 'i' } },
              { shortDescription: { $regex: escapedSearch, $options: 'i' } },
              { name: { $regex: escapedSearch, $options: 'i' } },
            ...(isNaN(escapedSearch) ? [] : [{ lotNumber: { $eq: escapedSearch } }])
            ]
          },
        ];
        
        const priceFilter: any = {};

        if (fromprice !== undefined) {
          priceFilter.$gte = fromprice;
        }
        if (toprice !== undefined) {
          priceFilter.$lte = toprice;
        }

        if (Object.keys(priceFilter).length > 0) {
          query.$and.push({ startPrice: priceFilter });
        }
        
        const materialFilter: any = {};
        
        if (material != null && material !== '') {
          // match the whole string, case-insensitive
          materialFilter.material = {
            $regex: `^${material}$`,
            $options: 'i'
          };
        
          query.$and.push(materialFilter);
        }
        
        const statusfilter: any = {};
        
        if (itemstatus != null && itemstatus !== '') {
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            if(itemstatus=='ongoing') {
                query.$and.push({ auctionStartDate: { $gte: today.toISOString().split('T')[0] } , auctionEndDate: {$lte: today.toISOString().split('T')[0] } });
            }   
            if(itemstatus=='upcoming') {
                query.$and.push({ auctionStartDate: { $gte: today.toISOString().split('T')[0] } });
            }    
            if(itemstatus=='ended') {
                query.$and.push({ auctionEndDate: { $lt: today.toISOString().split('T')[0] } });
            }  
        }
        
        
      const products = await this.productModel
        .find(query)
        .skip(skip)
        .sort({ lotNumber: 1 })
        .limit(limit).populate('category')
        .exec();

      const total = await this.productModel.countDocuments(query).exec();

      return {
        data: products,
        total,
        page,
        limit,
        totalPages: Math.ceil(total / limit),
      };

    } else {

      const query: any = {};

        var escapedSearch: any ='';
        if (search) {
          escapedSearch = search.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
          //query.sku = { $regex: escapedSearch, $options: 'i' },
        } 
        //query.sku = { $regex: escapedSearch, $options: 'i' },

        /*query.$or = [
          { sku: { $regex: escapedSearch, $options: 'i' } },
          { description: { $regex: escapedSearch, $options: 'i' } },
          { shortDescription: { $regex: escapedSearch, $options: 'i' } },
          { name: { $regex: escapedSearch, $options: 'i' } },
          { category: cats._id }
        ];*/

        const categoryId = cats._id;

        // Step 1: Find subcategories recursively
        const categories = await this.categoryModel
            .find({ $or: [{ _id: categoryId }, { parent: categoryId }] })
            .exec();

        const categoryIds = categories.map(category => category._id);

        query.$and = [
          {
            $or: [
              { sku: { $regex: escapedSearch, $options: 'i' } },
              { description: { $regex: escapedSearch, $options: 'i' } },
              { shortDescription: { $regex: escapedSearch, $options: 'i' } },
              { name: { $regex: escapedSearch, $options: 'i' } }
            ]
          },
          { category:  { $in: categoryIds } }
        ];
        
        
        
        const priceFilter: any = {};

        if (fromprice !== undefined) {
          priceFilter.$gte = fromprice;
        }
        if (toprice !== undefined) {
          priceFilter.$lte = toprice;
        }
        if (Object.keys(priceFilter).length > 0) {
          query.$and.push({ startPrice: priceFilter });
        }
        
        const materialFilter: any = {};
        
        if (material != null && material !== '') {
          // match the whole string, case-insensitive
          materialFilter.material = {
            $regex: `^${material}$`,
            $options: 'i'
          };
        
          query.$and.push(materialFilter);
        }
        
        const statusfilter: any = {};
        
        if (itemstatus != null && itemstatus !== '') {
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            if(itemstatus=='ongoing') {
                query.$and.push({ auctionStartDate: { $gte: today } , auctionEndDate: {$lte: today} });
            }   
            if(itemstatus=='upcoming') {
                query.$and.push({ auctionStartDate: { $lt: today } });
            }    
            if(itemstatus=='ended') {
                query.$and.push({ auctionStartDate: { $gt: today } });
            }  
        }
        
        


      const products = await this.productModel
        .find(query)
        .skip(skip)
        .limit(limit).populate('category')
        .exec();

      const total = await this.productModel.countDocuments(query).exec();

      return {
        data: products,
        total,
        page,
        limit,
        totalPages: Math.ceil(total / limit),
      };

    }

    
  }

  async deleteImage(body): Promise<any> {
    
    if(!body.image) {
      return {
        statuscode:'failed',
        message: 'Missing Image',
      };
    }
    await this.productModel.findByIdAndUpdate(
      body.id,
      { $pull: { images: body.image } },
      { new: true },
    ).exec();
    return {
      statuscode:'success',
      message: 'Image Removed from Product',
    };
  }

  /*async myBidHistory(body,id: string): Promise<any> {

      const result = await this.bidModel.aggregate([
        // Stage 1: Match bids by the user
        {
          $match: { user: id },
        },
        // Stage 2: Group by product and calculate the maximum bid
        {
          $group: {
            _id: '$product', // Group by product (string field)
            maxBid: { $max: '$bidAmount' }, // Calculate the maximum bid
          },
        },
        // Stage 3: Add a converted ObjectId field for matching
        {
          $addFields: {
            productId: { $toObjectId: '$_id' }, // Convert the string product ID to ObjectId
          },
        },
        // Stage 4: Lookup to fetch product details using ObjectId
        {
          $lookup: {
            from: 'products', // Name of the products collection
            localField: 'productId', // The converted ObjectId
            foreignField: '_id', // Match with the ObjectId in the products collection
            as: 'productDetails',
          },
        },
        // Stage 5: Unwind to simplify the productDetails array
        {
          $unwind: '$productDetails',
        },
        // Stage 6: Project to include all product details and the maximum bid
        {
          $project: {
            _id: 0, // Exclude the aggregation-generated _id field
            product: '$productDetails', // Include all fields from the product document
            maxBid: 1, // Include the maximum bid
          },
        },
      ]);

      //return result;
        return {
          statuscode:'success',
          data: result,
          id:id
        };

    
  }*/
  
  async myBidHistory(body,id: string): Promise<any> {
    
      const result = await this.bidModel.aggregate([
        // Stage 1: Match bids made by the logged-in user to find relevant products
        {
          $match: { user: id },
        },
        // Stage 2: Group by product to get distinct product IDs
        {
          $group: {
            _id: '$product', // Group by product (string field)
          },
        },
        // Stage 3: Add a converted ObjectId field for joining with products collection
        {
          $addFields: {
            productId: { $toObjectId: '$_id' }, // Convert the string product ID to ObjectId
          },
        },
        // Stage 4: Lookup to fetch product details using ObjectId
        {
          $lookup: {
            from: 'products', // Name of the products collection
            localField: 'productId', // The converted ObjectId
            foreignField: '_id', // Match with the ObjectId in the products collection
            as: 'productDetails',
          },
        },
        // Stage 5: Unwind productDetails to simplify the array
        {
          $unwind: '$productDetails',
        },
        // Stage 6: Lookup all bids for each product to determine the top bid across all users
        {
          $lookup: {
            from: 'bids', // Name of the bids collection
            localField: '_id', // Product ID (string) as stored in the bids collection
            foreignField: 'product', // Match with the product field in the bids collection
            as: 'allBids',
          },
        },
        // Stage 7: Find the maximum bid and the user who placed it
        {
          $addFields: {
            topBid: {
              $arrayElemAt: [
                {
                  $sortArray: { input: '$allBids', sortBy: { bidAmount: -1 } },
                },
                0,
              ],
            },
          },
        },
        // Stage 8: Project to include product details, maximum bid, and the user who placed it
        {
          $project: {
            _id: 0, // Exclude the aggregation-generated _id field
            product: '$productDetails', // Include all product details
            maxBid: '$topBid.bidAmount', // Include the maximum bid amount
            topBidUser: '$topBid.user', // Include the user who placed the maximum bid
          },
        },
      ]);

      //return result;
        return {
          statuscode:'success',
          data: result,
          id:id
        };
  }

  async myDashboard(body,id:string): Promise<any> {
    
    const count = await this.bidModel.aggregate([
      // Stage 1: Match bids by the user
      {
        $match: { user: id },
      },
      // Stage 2: Group by product
      {
        $group: {
          _id: '$product', // Group by product (ensures distinct products)
        },
      },
      // Stage 3: Count the number of distinct products
      {
        $count: 'productCount', // Returns the count of grouped products
      },
    ]);

    const wincount = await this.bidModel.aggregate([
      // Stage 1: Match bids by the user
      {
        $match: { user: id },
      },
      // Stage 2: Group by product (to ensure distinct products)
      {
        $group: {
          _id: '$product', // Group by product
        },
      },
      // Stage 3: Add a converted ObjectId field for matching
      {
        $addFields: {
          productId: { $toObjectId: '$_id' }, // Convert the product string ID to ObjectId
        },
      },
      // Stage 4: Lookup to join with products collection
      {
        $lookup: {
          from: 'products', // Name of the products collection
          localField: 'productId', // The converted ObjectId
          foreignField: '_id', // Match with the ObjectId in the products collection
          as: 'productDetails',
        },
      },
      // Stage 5: Filter products where finaluser matches the userId
      {
        $match: {
          'productDetails.finaluser': id,
        },
      },
      // Stage 6: Count the number of matching products
      {
        $count: 'productCount', // Returns the count of matching products
      },
    ]);
    
    const allbidcount = await this.bidModel.aggregate([
        // Stage 2: Group by product to get distinct product IDs
        {
          $group: {
            _id: '$product', // Group by product (string field)
          },
        },
        // Stage 3: Add a converted ObjectId field for joining with products collection
        {
          $addFields: {
            productId: { $toObjectId: '$_id' }, // Convert the string product ID to ObjectId
          },
        },
        // Stage 4: Lookup to fetch product details using ObjectId
        {
          $lookup: {
            from: 'products', // Name of the products collection
            localField: 'productId', // The converted ObjectId
            foreignField: '_id', // Match with the ObjectId in the products collection
            as: 'productDetails',
          },
        },
        // Stage 5: Unwind productDetails to simplify the array
        {
          $unwind: '$productDetails',
        },
        // Stage 6: Lookup all bids for each product to determine the top bid across all users
        {
          $lookup: {
            from: 'bids', // Name of the bids collection
            localField: '_id', // Product ID (string) as stored in the bids collection
            foreignField: 'product', // Match with the product field in the bids collection
            as: 'allBids',
          },
        },
        // Stage 7: Find the maximum bid and the user who placed it
        {
          $addFields: {
            topBid: {
              $arrayElemAt: [
                {
                  $sortArray: { input: '$allBids', sortBy: { bidAmount: -1 } },
                },
                0,
              ],
            },
          },
        },
        // Stage 8: Project to include product details, maximum bid, and the user who placed it
        {
          $project: {
            _id: 0, // Exclude the aggregation-generated _id field
            product: '$productDetails', // Include all product details
            maxBid: '$topBid.bidAmount', // Include the maximum bid amount
            topBidUser: '$topBid.user', // Include the user who placed the maximum bid
          },
        },
      ]);

    return {
      statuscode:'success',
      totalbids: count.length > 0 ? count[0].productCount : 0,
      winningbids: wincount.length > 0 ? wincount[0].productCount : 0,
      allbids: allbidcount.length,
    };

  }

  async getAdjacentProductsBySku(body): Promise<{ previous: any; current: any; next: any }> {
    // Step 1: Find the product with the given SKU
    var sku=body.sku;
    const currentProduct = await this.productModel.findOne({ sku }).exec();
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    if (!currentProduct) {
      throw new NotFoundException(`Product with SKU "${sku}" not found`);
    }

    const lotNumber = currentProduct.lotNumber;

    // Step 2: Find the previous product
    const previousProduct = await this.productModel
      .findOne({ lotNumber: { $lt: lotNumber }, auctionEndDate: { $gt: today }  }) // Find products with smaller lotNumber
      .sort({ lotNumber: -1 }) // Sort in descending order to get the closest smaller lotNumber
      .exec();

    // Step 3: Find the next product
    const nextProduct = await this.productModel
      .findOne({ lotNumber: { $gt: lotNumber}, auctionEndDate: { $gt: today }   }) // Find products with larger lotNumber
      .sort({ lotNumber: 1 }) // Sort in ascending order to get the closest larger lotNumber
      .exec();

    return {
      previous: previousProduct || null, // Return null if no previous product exists
      current: currentProduct, // Include the current product
      next: nextProduct || null, // Return null if no next product exists
    };
  }
  
  async allBidHistory(body,id: string): Promise<any> {
    
      /*const result = await this.bidModel.aggregate([
        // Stage 2: Group by product to get distinct product IDs
        {
          $group: {
            _id: '$product', // Group by product (string field)
          },
        },
        // Stage 3: Add a converted ObjectId field for joining with products collection
        {
          $addFields: {
            productId: { $toObjectId: '$_id' }, // Convert the string product ID to ObjectId
          },
        },
        // Stage 4: Lookup to fetch product details using ObjectId
        {
          $lookup: {
            from: 'products', // Name of the products collection
            localField: 'productId', // The converted ObjectId
            foreignField: '_id', // Match with the ObjectId in the products collection
            as: 'productDetails',
          },
        },
        // Stage 5: Unwind productDetails to simplify the array
        {
          $unwind: '$productDetails',
        },
        // Stage 6: Lookup all bids for each product to determine the top bid across all users
        {
          $lookup: {
            from: 'bids', // Name of the bids collection
            localField: '_id', // Product ID (string) as stored in the bids collection
            foreignField: 'product', // Match with the product field in the bids collection
            as: 'allBids',
          },
        },
        // Stage 7: Find the maximum bid and the user who placed it
        {
          $addFields: {
            topBid: {
              $arrayElemAt: [
                {
                  $sortArray: { input: '$allBids', sortBy: { bidAmount: -1 } },
                },
                0,
              ],
            },
          },
        },
        // Stage 8: Project to include product details, maximum bid, and the user who placed it
        {
          $project: {
            _id: 0, // Exclude the aggregation-generated _id field
            product: '$productDetails', // Include all product details
            maxBid: '$topBid.bidAmount', // Include the maximum bid amount
            topBidUser: '$topBid.user', // Include the user who placed the maximum bid
          },
        },
      ]);*/

      const newresult = await this.bidModel.aggregate([
        {
          $addFields: {
            productObjectId: { $toObjectId: "$product" } // Convert product field (string) to ObjectId
          }
        },
        {
          $group: {
            _id: "$productObjectId",  // Use the newly created ObjectId field for grouping
            highestBidAmount: { $max: "$bidAmount" }, // Get the highest bidAmount
            highestBidDetails: { $first: "$$ROOT" }  // Get the details of the highest bid
          }
        },
        {
          $lookup: {
            from: "products",           // Join with the 'products' collection
            localField: "_id",           // Use the ObjectId group field from the bids collection
            foreignField: "_id",         // Match the _id in the products collection (ObjectId)
            as: "productDetails"         // Name of the new field to store product details
          }
        },
        {
          $unwind: "$productDetails"    // Unwind the productDetails array to get the actual document
        }
      ]);

      //return result;
        return {
          statuscode:'success',
          data: newresult,
          id:id
        };
  }
  
  async requestVideo(body,user): Promise<any> {

    
    const productId=body.productid;
    const existingVideo = await this.videoModel.findOne({ user: user, product: productId });

    if (existingVideo) {
      return {
        statuscode:'failed',
        msg:'You have already requested video for this product. Please contact administrator if you have not recieved any updates.',
      };
    }

    // Create a new video entry
    const newVideo = new this.videoModel({
      user: user,
      product: productId,
    });

    await newVideo.save();

    return {
        statuscode:'success',
        msg:'Request for video for this product has been sent to Administrator.',
      };

  }

  async myrequestedVideo(body,user): Promise<any> {

    const existingVideos = await this.videoModel.find({ user: user }).populate('product');

    return {
        statuscode:'success',
        data:existingVideos,
      };

  }

  async allrequestedVideo(body,user): Promise<any> {

    const existingVideos = await this.videoModel.find().populate('product').populate('user');

    return {
        statuscode:'success',
        data:existingVideos,
      };

  }
  
    async alertFloorWinner(productId): Promise<any> {
      const productdata = await this.productModel.findById(productId).exec();
      if(productdata) {
        if(productdata.finaluser==null) {
          var plain = "Hello,  Congratulations! Bid for the item at the Travancore Cochin Heritage Auction has ended!. Item Details: " + productdata.name + ", Winning Bid: " + productdata.finalamount + ",Lot Number: " + productdata.lotNumber + ". Regards, Travancore Heritage Auctions .";
          var content = "<p>Hello ,</p><br> <p>Congratulations! Bid for the item at the Travancore Cochin Heritage Auction has ended!</p><br><br> <p>Item Details: " + productdata.name + ",</p><br> <p>Winning Bid: " + productdata.finalamount + ",</p><br> <p>Lot Number: " + productdata.lotNumber + "</p><br> <p>. Regards, Travancore Heritage Auctions .</p>";
          var subject = "Bidding Closed - Travancore Heritage Auction";
         // this.sendEmail("info@travancoreheritageauction.com","Admin",plain,content,subject);
        } else {
          const userdata = await this.userModel.findById(productdata.finaluser).exec();
          if(userdata) {
            var plain = "Hello " + userdata.firstname + ' ' + userdata.lastname + ", Congratulations! We are pleased to inform you that you have won the bid for the item at the Travancore Cochin Heritage Auction! Your successful bid has secured this remarkable piece. Thank you for your participation and enthusiasm. Item Details: " + productdata.name + ", Winning Bid: " + productdata.finalamount + ",Lot Number: " + productdata.lotNumber + ". Regards, Travancore Heritage Auctions .";
            var content = "<p>Hello " + userdata.firstname + ' ' + userdata.lastname + ",</p><br> <p>Congratulations! We are pleased to inform you that you have won the bid for the item at the Travancore Cochin Heritage Auction! Your successful bid has secured this remarkable piece. Thank you for your participation and enthusiasm.</p><br><br> <p>Item Details: " + productdata.name + ",</p><br> <p>Winning Bid: " + productdata.finalamount + ",</p><br> <p>Lot Number: " + productdata.lotNumber + "</p><br> <p>. Regards, Travancore Heritage Auctions .</p>";
            var subject = "Bidding Won - Travancore Heritage Auction";
            this.sendEmail("info@travancoreheritageauction.com",userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
            //this.sendEmail(userdata.email,userdata.firstname + ' ' + userdata.lastname,plain,content,subject);
          
            /*const result = await axios.post(
              'https://graph.facebook.com/v23.0/709680682221149/messages',
              {
                messaging_product: 'whatsapp',
                to: userdata.phone,
                type: 'template',
                template: {
                  name: 'sun_bidding_won_client',
                  language: {
                    code: 'en'
                  },
                  components: [
                    {
                      type: 'body',
                      parameters: [
                        {
                          type: 'text',
                          parameter_name:'name',
                          text: userdata.firstname + ' ' + userdata.lastname
                        },
                        {
                          type: 'text',
                          parameter_name:'itemname',
                          text: productdata.name
                        },
                        {
                          type: 'text',
                          parameter_name:'winningbid',
                          text: productdata.finalamount
                        },
                        {
                          type: 'text',
                          parameter_name:'lotnumber',
                          text: productdata.lotNumber
                        }
                      ]
                    }
                  ]
                }
              },
              {
                headers: {
                  'Content-Type': 'application/json',
                  Authorization: 'Bearer EAAFVXw8ns9ABO7hGWIU57K1PZBOlvDtylqUt8AfVF5AXt5i8aTZAa2qeqcVpE9eorejrLggCrIgBau77usFS7CZAZAtXxyl2RK9Mh4Qhw4ZBPUq6s4ArkvM8VfPVnNm58ua6jXhOTzhfqCcrSzPzpjpSniq36tGKICBxqEjJfqSfK68Vqbb0stCuORwOd6kGEHGCPFHOIQlsiC9R2FDXdGHxpZBZBXTZCEcyrU4nw1wLugnjS5T2OtvIhjYZB00EiZAV6I2sjsEuvC5QZDZD'
                }
              }
            );*/
          }
        }
      }
  }
  
    

  async getCurrentAndNextLot(): Promise<any> {
    
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    var currentLot = await this.productModel
    .findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 }, // auctionStatus exists and is 0
            { auctionStatus: { $exists: false } }, // auctionStatus field does not exist
          ],
        },
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $gt: 0 } },
        { auctionEndDate: { $gt: today } },
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec();

    /*var currentLot = await this.productModel
    .findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 }, // auctionStatus exists and is 0
            { auctionStatus: { $exists: false } }, // auctionStatus field does not exist
          ],
        },
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $gt: 1000 } }, // lotnumber exists and is not null
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec();*/

    if(currentLot && currentLot.finaluser) {
      var currentuserdata = await this.userModel.findById(currentLot.finaluser).exec();
    }

    const nextLot = currentLot ? await this.productModel
    .findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 },
            { auctionStatus: { $exists: false } },
          ],
        },
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $gt: currentLot.lotNumber } }, // lotnumber greater than the current lot
        { auctionEndDate: { $gt: today } },
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec() : null;

    if(nextLot && nextLot.finaluser) {
      var nextuserdata = await this.userModel.findById(nextLot.finaluser).exec();
    }



    var previousLot = currentLot ? await this.productModel
    .findOne({
      $and: [
        /*{
          $or: [
            { auctionStatus: 0 },
            { auctionStatus: { $exists: false } },
          ],
        },*/
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $lt: currentLot.lotNumber } }, // lotnumber greater than the current lot
        { auctionEndDate: { $gt: today } },
      ],
    })
    .sort({ lotNumber: -1 }) // Sort by lotnumber in ascending order
    .exec() : null;

    if(previousLot && previousLot.finaluser) {
      var previoususerdata = await this.userModel.findById(previousLot.finaluser).exec();
    }
    
    if(!previousLot) {
      
      var previousLot = await this.productModel
      .findOne()
      .sort({ lotNumber: -1 }) // Sort by lotnumber in ascending order
      .exec();
    
    } 

    var lastbid = await this.bidModel.find({ product: previousLot?._id.toString() }).exec();    

    return {
      current: currentLot,
      currentuser: currentuserdata,
      next: nextLot,
      nextuser: nextuserdata,
      previous: previousLot,
      previoususer: previoususerdata,
      previousbid: lastbid,
    };

  }
  
  async gethomeCurrentAndNextLot(): Promise<any> {
    
    var currentLot = await this.productModel
    .findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 }, // auctionStatus exists and is 0
            { auctionStatus: { $exists: false } }, // auctionStatus field does not exist
          ],
        },
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $gt: 0 } },
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec();

    /*var currentLot = await this.productModel
    .findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 }, // auctionStatus exists and is 0
            { auctionStatus: { $exists: false } }, // auctionStatus field does not exist
          ],
        },
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $gt: 1000 } }, // lotnumber exists and is not null
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec();*/

    if(currentLot && currentLot.finaluser) {
      var currentuserdata = await this.userModel.findById(currentLot.finaluser).exec();
    }

    const nextLot = currentLot ? await this.productModel
    .findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 },
            { auctionStatus: { $exists: false } },
          ],
        },
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $gt: currentLot.lotNumber } }, // lotnumber greater than the current lot
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec() : null;

    if(nextLot && nextLot.finaluser) {
      var nextuserdata = await this.userModel.findById(nextLot.finaluser).exec();
    }



    const previousLot = currentLot ? await this.productModel
    .findOne({
      $and: [
        /*{
          $or: [
            { auctionStatus: 0 },
            { auctionStatus: { $exists: false } },
          ],
        },*/
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotNumber: { $lt: currentLot.lotNumber } }, // lotnumber greater than the current lot
      ],
    })
    .sort({ lotNumber: -1 }) // Sort by lotnumber in ascending order
    .exec() : null;

    if(previousLot && previousLot.finaluser) {
      var previoususerdata = await this.userModel.findById(previousLot.finaluser).exec();
    }

    return {
      current: currentLot,
      currentuser: currentuserdata,
      next: nextLot,
      nextuser: nextuserdata,
      previous: previousLot,
      previoususer: previoususerdata
    };

  }

  async generateResult(body,user): Promise<any> {
    var productId = body.productid;
    const product = await this.productModel.findById(body.productid).exec();
    
    
    const currentTime = new Date();
    
    if (product.lotstart == '1') {
      const extendAuctionForSeconds = product.extendAuctionForSeconds || 20;

      // Find the last bid for the product
      const lastBid = await this.bidModel.find({ product: productId })
      .sort({ createdAt: -1 }) // Sort by createdAt in descending order
      .limit(1)
      .exec();
      
       
       if (lastBid.length > 0) {
        const lastBidTime = new Date(lastBid[0].createdAt);
        const timeDifference = (currentTime.getTime() - lastBidTime.getTime()) / 1000; // Difference in seconds
        
        if (timeDifference > extendAuctionForSeconds) {
           
           if(lastBid[0].bidAmount<product.reservePrice) {
            /* Bid less than reserve Price */
            
            await this.productModel.findByIdAndUpdate(
              productId,
              { auctionStatus: '-1' },
            );
            await this.alertFloorWinner(productId);
            const newResult = new this.resultModel({
              product: productId,
              auctionStatus:'-1'
            });
            await newResult.save();
            return {
              statuscode:'failure',
              message: 'Auction has ended without being sold',
            };
          } else {
            
            /* Check Reserve Price */
            var highestReserve = await this.reserveModel
              .findOne({ product: productId })
              .sort({ reserveAmount: -1 })
              .select('user reserveAmount')
              .exec();
             
            if(highestReserve) {
                
                if(lastBid[0].bidAmount<=highestReserve.reserveAmount && highestReserve.user == lastBid[0].user)
                {
                    
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { finaluser: lastBid[0].user },
                    );
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { finalamount: lastBid[0].bidAmount },
                    );
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { auctionStatus: '1' },
                    );
                }
                if(lastBid[0].bidAmount<=highestReserve.reserveAmount && highestReserve.user != lastBid[0].user)
                {
                    const setting = await this.settingModel.findOne({
                      fromamount: { $lte: lastBid[0].bidAmount },
                      toamount: { $gte: lastBid[0].bidAmount },
                    });
            
                    const bidIncrement = setting ? setting.bidincrement : 0;
            
                    if(lastBid.length > 0) {
                      var nextBidAmount = lastBid[0].bidAmount + bidIncrement;
                    } else {
                      var nextBidAmount = lastBid[0].bidAmount;
                    }
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { finaluser: highestReserve.user },
                    );
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { finalamount: nextBidAmount },
                    );
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { auctionStatus: '1' },
                    );
                    /*await this.newBid({productid:body.productid,amount:nextBidAmount},highestReserve.user);
                    return {
                      statuscode:'success',
                      message: `Bid has changed`,
                    };*/
                    
                }
                if(lastBid[0].bidAmount>highestReserve.reserveAmount)
                {
                   
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { finaluser: lastBid[0].user },
                    );
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { finalamount: lastBid[0].bidAmount },
                    );
                    await this.productModel.findByIdAndUpdate(
                      productId,
                      { auctionStatus: '1' },
                    );
                }
            } else {
               console.log("got here");
                await this.productModel.findByIdAndUpdate(
                  productId,
                  { finaluser: lastBid[0].user },
                );
                await this.productModel.findByIdAndUpdate(
                  productId,
                  { finalamount: lastBid[0].bidAmount },
                );
                await this.productModel.findByIdAndUpdate(
                  productId,
                  { auctionStatus: '1' },
                );
            }
            
            
            //await this.alertFloorWinner(productId);
            const newResult = new this.resultModel({
              product: productId,
              auctionStatus:'1'
            });
            await newResult.save();
            return {
              statuscode:'success',
              message: `Bidding has ended for this product and result declared`,
            };
          }
        } else {
          
          return {
            statuscode:'failure',
            message: 'Bid Result awaiting as bid extension time is not over',
          };
        }
      } else {
        const lastBidTime = new Date(product.productDate);
        const timeDifference = (currentTime.getTime() - lastBidTime.getTime()) / 1000; // Difference in seconds

        if (timeDifference > extendAuctionForSeconds) {
          await this.productModel.findByIdAndUpdate(
            productId,
            { auctionStatus: '-1' },
          );
          await this.alertFloorWinner(productId);
            const newResult = new this.resultModel({
              product: productId,
              auctionStatus:'-1'
            });
            await newResult.save();
          return {
            statuscode:'failure',
            message: 'Auction has ended without being sold',
          };
        } else {

          return {
            statuscode:'failure',
            message: 'Bid Result awaiting as bid extension time is not over',
          };
        }
      }
      
    } 
    /*else {
        var lastBid = await this.bidModel.find({ product: productId })
        .sort({ createdAt: -1 }) // Sort by createdAt in descending order
        .limit(1)
        .exec();
          
        if (lastBid.length > 0) {

          if(lastBid[0].bidAmount<product.reservePrice) {
          
            await this.productModel.findByIdAndUpdate(
                  productId,
                  { auctionStatus: '-1' },
                );
                await this.alertFloorWinner(productId);
                const newResult = new this.resultModel({
                  product: productId,
                  auctionStatus:'-1'
                });
                await newResult.save();
                return {
                  statuscode:'failure',
                  message: 'Auction has ended without being sold',
                };
                
            } else {
            
              await this.productModel.findByIdAndUpdate(
                  productId,
                  { finaluser: lastBid[0].user },
                );
                await this.productModel.findByIdAndUpdate(
                  productId,
                  { finalamount: lastBid[0].bidAmount },
                );
                await this.productModel.findByIdAndUpdate(
                  productId,
                  { auctionStatus: '1' },
                );
                await this.alertFloorWinner(productId);
                const newResult = new this.resultModel({
                  product: productId,
                  auctionStatus:'1'
                });
                await newResult.save();
                return {
                  statuscode:'success',
                  message: `Bidding has ended for this product and result declared`,
                };
            
            }
            
        } else {
        
            await this.productModel.findByIdAndUpdate(
                productId,
                { auctionStatus: '-1' },
            );
            await this.alertFloorWinner(productId);
            const newResult = new this.resultModel({
              product: productId,
              auctionStatus:'-1'
            });
            await newResult.save();
            return {
                statuscode:'failure',
                message: 'Auction has ended without being sold',
            };
        }
    }*/
  }
  
  async getActiveLot(): Promise<any> {

    var currentLot = await this.productModel.findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 }, // auctionStatus exists and is 0
            { auctionStatus: { $exists: false } }, // auctionStatus field does not exist
          ],
        },
        { lotNumber: { $exists: true, $ne: null } }, // lotnumber exists and is not null
        { lotstart: 1 },
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec();

    return {
      current: currentLot,
    };

  }
  
  async getNextLiveLot(): Promise<any> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    var currentLot = await this.productModel.findOne({
      $and: [
        {
          $or: [
            { auctionStatus: 0 }, // auctionStatus exists and is 0
            { auctionStatus: { $exists: false } }, // auctionStatus field does not exist
          ],
        },
        { lotNumber: { $exists: true, $ne: null, $gt:0 } }, // lotnumber exists and is not null
        { lotstart: 1 },
        { auctionEndDate: {$gt:today}},
      ],
    })
    .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
    .exec();
    
    if(currentLot) {
        return {
          current: currentLot,
        };
    } else {
        var upcomingLot = await this.productModel.findOne({
          $and: [
            {
              $or: [
                { auctionStatus: 0 }, // auctionStatus exists and is 0
                { auctionStatus: { $exists: false } }, // auctionStatus field does not exist
              ],
            },
            { lotNumber: { $exists: true, $ne: null, $gt:0 } }, // lotnumber exists and is not null
            { lotstart: 0 },
            { auctionEndDate: {$gt:today}},
          ],
        })
        .sort({ lotNumber: 1 }) // Sort by lotnumber in ascending order
        .exec();
        
        return {
          current: upcomingLot,
        };
    }

    

  }

  async lastResult(): Promise<any> {

    const result = await this.resultModel
      .findOne()
      .sort({ createdAt: -1 })  // Sort by createdAt in descending order (most recent first)
      .populate('product')  // Populate the 'product' field with the Product document
      .exec();

    return {
      result: result,
    };

  }

  /* Cart Code */

  async getCart(userId: string): Promise<Cart | null> {
    return this.cartModel.findOne({ user: new Types.ObjectId(userId) }).populate('items.productId').exec();
  }

  async addToCart(userId: string, productId: string, quantity: number): Promise<any> {
    if (!productId || !quantity) {
      throw new Error('Product ID and quantity are required');
    }

    let cart = await this.cartModel.findOne({ user: new Types.ObjectId(userId) });

    if (cart) {
      // Ensure 'items' array exists
      if (!cart.items) {
        cart.items = [];
      }

      const itemIndex = cart.items.findIndex((item) => item.productId.toString() === productId);

      if (itemIndex > -1) {
        // Update quantity if product already exists
        //cart.items[itemIndex].quantity += quantity;
        return {
          statuscode:'failure',
          message: 'You have already added this product in to the cart.',
          cart:cart
        };
      } else {
        // Add new product if it doesn't exist
        cart.items.push({ productId: new Types.ObjectId(productId), quantity });
        await cart.save();
      }

      
      return {
        statuscode:'success',
        message: 'Product added in to the cart.',
        cart:await this.cartModel.findOne({ user: new Types.ObjectId(userId) })
      };
    }

    // If no cart exists, create a new one
    const newCart = new this.cartModel({
      user: new Types.ObjectId(userId),
      items: [{ productId: new Types.ObjectId(productId), quantity }],
    });

    await newCart.save();
    return {
      statuscode:'success',
      message: 'Product added in to the cart.',
      cart:await this.cartModel.findOne({ user: new Types.ObjectId(userId) })
    };
  }

  async removeFromCart(userId: string, productId: string): Promise<any> {
    const cart = await this.cartModel.findOne({ user: new Types.ObjectId(userId) });
    if (cart) {
      cart.items = cart.items.filter((item) => item.productId.toString()  !== productId);
      await cart.save();
      return {
        statuscode:'success',
        message: 'Product removed from cart.',
        cart:await this.cartModel.findOne({ user: new Types.ObjectId(userId) })
      };
    }
  }

  async clearCart(userId: string): Promise<any> {
    await this.cartModel.findOneAndUpdate({ user: new Types.ObjectId(userId) }, { items: [] });
    return {
        statuscode:'success',
        message: 'Cart Cleared.',
        cart:await this.cartModel.findOne({ user: new Types.ObjectId(userId) })
      };
  }

  async isProductInCart(userId: string, productId: string): Promise<{ exists: boolean }> {
    const cart = await this.cartModel.findOne({ user: new Types.ObjectId(userId) });
    
    if (!cart) {
      return { exists: false };
    }

    const productExists = cart.items.some((item) => item.productId.toString() === productId);
    return { exists: productExists };
  }

  /* Cart Code Ends */
  
    async changeManulStatus(body): Promise<any> {
        const objectId = new Types.ObjectId(body.id); // Ensure it's a valid ObjectId
        const filter = { _id: objectId };
        const update = { $set: { auctionStatus: body.val } };
    
        const result = await this.productModel.updateOne(filter, update);
        return result;
    }
}