import {  Controller,  Get, Req, Post,  Query,  Body,  Param,  Put,  Delete,  UploadedFiles,  UseInterceptors,  UseGuards,  BadRequestException, UsePipes, ValidationPipe} from '@nestjs/common';
import { FilesInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { SettingsService } from './settings.service';
import { SettingsDto } from './settings.dto';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
import { Setting, SettingSchema } from './setting.schema';

@Controller('settings') // Base route for all user endpoints
export class SettingsController {
  constructor(private readonly settingsService: SettingsService) {}

  @UseGuards(JwtAuthGuard)
  @Post()
  @UsePipes(new ValidationPipe({ transform: true }))
  async createBidIncrement(@Body() body: any, settingsDto: SettingsDto,) {
    //const transformed = new settingsDto();
    //Object.assign(transformed, body);
    console.log(body);
    return this.settingsService.createIncrement(body);
  }

  //@UseGuards(JwtAuthGuard)
  @Get()
  async getIncrements(): Promise<Setting[]> {
    return this.settingsService.findAll();
  }

  @UseGuards(JwtAuthGuard)
  @Delete(':id')
  async delete(@Param('id') id: string): Promise<any> {
    return this.settingsService.delete(id);
  }

  // Update a user by ID
  @UseGuards(JwtAuthGuard)
  @Post('banner')
  @UseInterceptors(
    FilesInterceptor('images', 1, { // Match `images[]` instead of just `images`
      storage: diskStorage({
        destination: './uploads',
        filename: (req, file, cb) => {
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
          const ext = extname(file.originalname);
          cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
        },
      }),
      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png|giff)$/)) {
          return cb(new BadRequestException('Only image files are allowed!'), false);
        }
        cb(null, true);
      },
    }),
  )
  async addBanner(@Body() body: any,@Req() req, @UploadedFiles() files: Express.Multer.File[]) {
    const newImages = files && files.length > 0 ? files.map((file) => file.filename) : [];
    return this.settingsService.addBanner(body,req.user.userId, newImages);
  }

  @Get('banners')
  async getBanners(): Promise<Setting[]> {
    return this.settingsService.getBanners();
  }

  @UseGuards(JwtAuthGuard)
  @Post('delete-banner')
  async removeBanner(@Body() body: any,@Req() req): Promise<void> {
    return this.settingsService.removeBanner(body);
  }
}
