import {  Controller,  Get, Req, Post,  Query,  Body,  Param,  Put,  Delete,  UploadedFiles,  UseInterceptors,  UseGuards,  BadRequestException, UsePipes, ValidationPipe} from '@nestjs/common';
import { FilesInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { UsersService } from './users.service';
import { CreateUserDto } from './dto/create-user.dto';
import { LoginUserDto } from './dto/login-user.dto';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';

@Controller('users') // Base route for all user endpoints
export class UsersController {
  constructor(private readonly usersService: UsersService) {}

  @UseGuards(JwtAuthGuard)
  @Post()
  async createUser(@Body() createUserDto: any) {
    return this.usersService.createUser(createUserDto);
  }

  @UseGuards(JwtAuthGuard)
  @Get('dashboard')
  getDashboard(@Req() req) {
    const user = req.user; // user object extracted from token
    return {
      status: 'success',
      message: 'Welcome to the dashboard',
      user,
    };
  }

  @Post('register')
  async registerUser(@Body() createUserDto: CreateUserDto) {
    console.log(createUserDto);
   const userstatus = await this.usersService.createUser(createUserDto);
   return {
      status: 'success',
      message: 'Your account has been created and login and OTP details has been shared to your email. Please Login and use the OTP to verify your account and start bidding.',
    };
  }

  @Post('resend-otp')
  async resendOTP(@Body() body) {
   const userstatus = await this.usersService.resendOTP(body);
   return {
      status: 'success',
      message: 'Your OTP details has been shared to your email. Please Login and use the OTP to verify your account and start bidding.',
    };
  }

  @UseGuards(JwtAuthGuard)
  @Get()
  async getAllUsers() {
    return this.usersService.getAllUsers();
  }

  @UseGuards(JwtAuthGuard)
  @Post('inactive')
  async getInactiveUsers() {
    return await this.usersService.getInactiveUsers();
  }

  @UseGuards(JwtAuthGuard)
  @Post('approve/:id')
  async updateInactiveUsers(@Param('id') id: string) {
    const updatedUser = await this.usersService.approveUser(id);
    if (!updatedUser) {
      return { message: 'User not found or already approved' };
    }
    return { message: 'User approved successfully', user: updatedUser };
  }

  // Get a user by ID
  @Get(':id')
  async getUserById(@Param('id') id: string) {
    return this.usersService.getUserById(id);
  }

  // Update a user by ID
  @Put(':id')
  @UseInterceptors(
    FilesInterceptor('images', 10, { // Match `images[]` instead of just `images`
      storage: diskStorage({
        destination: './uploads',
        filename: (req, file, cb) => {
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
          const ext = extname(file.originalname);
          cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
        },
      }),
      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png|gif|doc|docx|pdf|xls|xlsx)$/)) {
          return cb(new BadRequestException(file.mimetype + 'Only image/doc and pdf files are allowed!'), false);
        }
        cb(null, true);
      },
    }),
  )
  async updateUser(@Param('id') id: string, @Body() updateUserDto: any, @UploadedFiles() files: Express.Multer.File[]) {
    /*if (files && files.length > 0) {
      updateUserDto.idproof = files.map((file) => file.filename); // Save file names
    }*/
    updateUserDto.dob  = updateUserDto.dob ? new Date(updateUserDto.dob) : null;
    const newImages = files && files.length > 0 ? files.map((file) => file.filename) : [];
    return this.usersService.updateUser(id, updateUserDto, newImages);
  }

  @UseGuards(JwtAuthGuard)
  @Post('delete-my-document')
  @UseInterceptors(FilesInterceptor(''))
  async deleteDocs(@Body() body: any,@Req() req) {
    return await this.usersService.deleteDocs(body);
  }

  // Delete a user by ID
  @Delete(':id')
  async deleteUser(@Param('id') id: string) {
    return this.usersService.deleteUser(id);
  }

  @Post('login')
  async login(@Body() loginUserDto: LoginUserDto) {
    return await this.usersService.login(loginUserDto);
  }

  @UseGuards(JwtAuthGuard)
  @Post('profile')
  async getProfile(@Req() req) {
    //const user = req.user; // user object extracted from token
    const user = await this.usersService.getUserById(req.user.userId);
    return {
      status: 'success',
      message: 'User Data',
      userdata: user,
    };
  }

  @UseGuards(JwtAuthGuard)
  @Post('update-maxbid')
  async updateBidAmount(@Body() body) {
    return await this.usersService.updateBidAmount(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('change-password')
  async changePassword(@Body() body) {
    return await this.usersService.changePassword(body);
  }

  @Post('reset-password')
  async resetPassword(@Body() body) {
    return await this.usersService.resetPassword(body);
  }
  

  @UseGuards(JwtAuthGuard)
  @Post('email')
  async send() {
    return await this.usersService.send();
  }

  @UseGuards(JwtAuthGuard)
  @Post('new-data')
  async createData(@Body() body: any,@Req() req) {
    return this.usersService.createData(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('get-data')
  async getData(@Body() body: any,@Req() req): Promise<any> {
    return this.usersService.getData(req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('delete-data')
  async delete(@Body() body: any,@Req() req): Promise<any> {
    return this.usersService.deleteData(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('get-user-data')
  async getUserData(@Body() body: any,@Req() req): Promise<any> {
    return this.usersService.getUserData(body);
  }

  @UseGuards(JwtAuthGuard)
  @Post('usercount')
  async getTotalUsersCount(): Promise<{ count: number }> {
    return await this.usersService.getTotalUsersCount();
  }
}
