import {  Controller,  Get, Req, Post,  Query,  Body,  Param,  Put,  Delete,  UploadedFiles,  UseInterceptors,  UseGuards,  BadRequestException, UsePipes, ValidationPipe} from '@nestjs/common';
import { FilesInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { ProductService } from './product.service';
import { Product } from './schemas/product.schema';
import { ProductDto } from './dto/product.dto';
import { extname } from 'path';
import { JwtAuthGuard } from '../auth/jwt-auth.guard';
const mongoose = require('mongoose');

@Controller('products')
export class ProductController {
  constructor(private readonly productService: ProductService) {}

  @UseGuards(JwtAuthGuard)
  @Post()
  @UsePipes(new ValidationPipe({ transform: true }))
  @UseInterceptors(
    FilesInterceptor('images', 5, { // Match `images[]` instead of just `images`
      storage: diskStorage({
        destination: './uploads',
        filename: (req, file, cb) => {
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
          const ext = extname(file.originalname);
          cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
        },
      }),
      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png|gif)$/)) {
          return cb(new BadRequestException('Only image files are allowed!'), false);
        }
        cb(null, true);
      },
    }),
  )
  async create(@Body() body: any, productDto: ProductDto, @UploadedFiles() files: Express.Multer.File[]): Promise<Product> {

    const transformed = new ProductDto();
    Object.assign(transformed, body); // Manually transform for debugging
    
    // Handle the uploaded files
    if (files && files.length > 0) {
      transformed.images = files.map((file) => file.filename); // Save file names
    }
    transformed.category = new mongoose.Types.ObjectId( transformed.category);
    console.log('Transformed DTO:', transformed);
    transformed.auctionStartDate= transformed.auctionStartDate ? new Date(transformed.auctionStartDate) : null;
    transformed.auctionEndDate= transformed.auctionEndDate ? new Date(transformed.auctionEndDate) : null;

    return this.productService.create(transformed);
  }

  @UseGuards(JwtAuthGuard)
  @Post('all')
  async findAll(): Promise<Product[]> {
    console.log("here");
    return this.productService.findAll();
  }

  @Get(':id')
  async findOne(@Param('id') id: string): Promise<Product> {
    return this.productService.findOne(id);
  }

  @UseGuards(JwtAuthGuard)
  @Put(':id')
  @UsePipes(new ValidationPipe({ transform: true }))
  @UseInterceptors(
    FilesInterceptor('images', 5, { // Match `images[]` instead of just `images`
      storage: diskStorage({
        destination: './uploads',
        filename: (req, file, cb) => {
          const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
          const ext = extname(file.originalname);
          cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
        },
      }),
      fileFilter: (req, file, cb) => {
        if (!file.mimetype.match(/\/(jpg|jpeg|png|gif)$/)) {
          return cb(new BadRequestException('Only image files are allowed!'), false);
        }
        cb(null, true);
      },
    }),
  )
  async update(@Param('id') id: string, @Body() body: any, productDto: ProductDto, @UploadedFiles() files: Express.Multer.File[]): Promise<Product> {
    
    const transformed = new ProductDto();
    Object.assign(transformed, body); // Manually transform for debugging
    
    // Handle the uploaded files
    if (files && files.length > 0) {
      transformed.images = files.map((file) => file.filename); // Save file names
    }
    transformed.category = new mongoose.Types.ObjectId( transformed.category);
    console.log('Transformed DTO:', transformed);
    transformed.auctionStartDate= transformed.auctionStartDate ? new Date(transformed.auctionStartDate) : null;
    transformed.auctionEndDate= transformed.auctionEndDate ? new Date(transformed.auctionEndDate) : null;

    return this.productService.update(id, transformed);
  }

  @UseGuards(JwtAuthGuard)
  @Delete(':id')
  async remove(@Param('id') id: string): Promise<void> {
    return this.productService.remove(id);
  }

  @Post('categories-with-count')
  async getCategoriesWithProductCount() {
    return this.productService.getCategoriesWithProductCount();
  }

  @Post('categories')
  @UseInterceptors(FilesInterceptor(''))
  async getProductsByCategory(@Body() body: any) {
    return this.productService.getProductsByCategory(body.name);
  }

  @Post('sku')
  @UseInterceptors(FilesInterceptor(''))
  async getProductBySKU(@Body() body: any) {
    return this.productService.findBySKU(body.sku);
  }

  @UseGuards(JwtAuthGuard)
  @Post('new-bid')
  @UseInterceptors(FilesInterceptor(''))
  async postNewBid(@Body() body: any,@Req() req) {
    return this.productService.newBid(body,req.user.userId);
  }

  @UseGuards(JwtAuthGuard)
  @Post('get-bid')
  @UseInterceptors(FilesInterceptor(''))
  async getBidHistory(@Body() body: any,@Req() req) {
    return this.productService.bidHistory(body);
  }


}
