import { Injectable, ConflictException, InternalServerErrorException, NotFoundException } from '@nestjs/common';
import { InjectModel } from '@nestjs/mongoose';
import { Model } from 'mongoose';
import { Category } from '../categories/schemas/category.schema';
import { Product } from './schemas/product.schema';
import { Bid } from './schemas/bid.schema';
import { Setting } from '../settings/setting.schema';
import { User } from '../users/schemas/user.schema';
import { ProductDto } from './dto/product.dto';

@Injectable()
export class ProductService {
  constructor(
    @InjectModel('Category') private readonly categoryModel: Model<Category>,
    @InjectModel(Product.name) private productModel: Model<Product>,
    @InjectModel(User.name) private userModel: Model<User>,
    @InjectModel(Bid.name) private bidModel: Model<Bid>,
    @InjectModel(Setting.name) private settingModel: Model<Setting>
  ) {}

  async create(productDto: Partial<Product>): Promise<Product> {
    try {
      const newProduct = new this.productModel(productDto);
      return await newProduct.save();
    } catch (error) {
      if (error.code === 11000) { // MongoDB duplicate key error code
        throw new ConflictException('Product SKU already exists.');
      }
      throw new InternalServerErrorException('Something went wrong');
    }
  }

  async findAll(): Promise<Product[]> {
    return this.productModel.find().populate('category').exec();
  }

  async findOne(id: string): Promise<Product> {
    const product = await this.productModel.findById(id).populate('category').exec();
    if (!product) {
      throw new NotFoundException(`Product with id ${id} not found`);
    }
    return product;
  }

  async update(id: string, productDto: any): Promise<Product> {
    const updatedProduct = await this.productModel.findByIdAndUpdate(id, productDto, { new: true }).exec();
    if (!updatedProduct) {
      throw new NotFoundException(`Product with id ${id} not found`);
    }
    return updatedProduct;
  }

  async remove(id: string): Promise<void> {
    const result = await this.productModel.findByIdAndDelete(id).exec();
    if (!result) {
      throw new NotFoundException(`Product with id ${id} not found`);
    }
  }

  async getCategoriesWithProductCount(): Promise<any[]> {
    return this.categoryModel.aggregate([
      {
        $lookup: {
          from: 'products', // The collection name for products
          localField: '_id', // The field in Category to match
          foreignField: 'category', // The field in Product to match
          as: 'products', // The resulting array of matched products
        },
      },
      {
        $addFields: {
          productCount: { $size: '$products' }, // Count the number of products
        },
      },
      {
        $project: {
          products: 0, // Exclude the full product list if not needed
        },
      },
    ]).exec();
  }

  async getProductsByCategory(name): Promise<any[]> {
    try {
      if(!name) {
        return this.productModel.find().populate('category').exec();
      } else {
        const category = await this.categoryModel.findOne({ name: name }).exec();
        
        if (!category) {
          console.log(`Category with name "${name}" not found.`);
          return [];
        }

        const products = await this.productModel.find({ category: category._id }).populate('category');

        return products;
      }
    } catch (error) {
      console.error('Error fetching products by category name:', error);
      throw error;
    }
  }

  async findBySKU(sku): Promise<Product> {
    return this.productModel.findOne({ sku }).populate('category').exec();
  }

  async findByLots(): Promise<any> {
    return this.productModel.find({ lotNumber: { $ne : null } }).populate('category').sort({ lotNumber: 1 }).exec();
  }

  async bidHistory(body): Promise<any> {

    const product = await this.productModel.findById(body.productid).exec();
    if(product) {
      const startPrice=product.startPrice ? product.startPrice : 1;
      const maxBid = await this.bidModel
      .find({ product: body.productid })
      .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
      .limit(1) // Get the highest bid
      .exec();

      const currentMaxBidAmount = maxBid.length > 0 ? maxBid[0].bidAmount : startPrice;
      const currentMaxBidUser = maxBid.length > 0 ? maxBid[0].user : '';

      const setting = await this.settingModel.findOne({
        fromamount: { $lte: currentMaxBidAmount },
        toamount: { $gte: currentMaxBidAmount },
      });

      const bidIncrement = setting ? setting.bidincrement : 0;

      const nextBidAmount = currentMaxBidAmount + bidIncrement;

      return {
        statuscode:'success',
        currentBid: currentMaxBidAmount,
        nextBid:nextBidAmount,
        currentBidUser:currentMaxBidUser,
        bid: await this.bidModel.find({ product: body.productid }).populate('user').sort({ bidAmount: -1 }).exec(),
      };

    } else {
      return {
        statuscode:'failure',
        message: 'Invalid Product Request',
      };
    }
  }

  async newBid(body,user): Promise<any> {
    const userdata = await this.userModel.findById(user).exec();
    const usermaxbidamount=userdata.maxBidAmount ? userdata.maxBidAmount : 0;
    const productId=body.productid;
    if(userdata.status==0) {
      //user bidlimit not yet approved by admin
      return {
        statuscode:'failure',
        message: 'Your maximum bid limit is yet to be approved by admin. Please contact administrator',
      };
    } else {
      const product = await this.productModel.findById(body.productid).exec();
      if(product) {
        const startPrice=product.startPrice ? product.startPrice : 1;

        const currentTime = new Date();
        if (currentTime > product.auctionEndDate) {
          return {
            statuscode:'failure',
            message: 'Bidding time is over',
          };
        }

        const maxBid = await this.bidModel
        .find({ product: body.productid })
        .sort({ bidAmount: -1 }) // Sort by bidAmount in descending order
        .limit(1) // Get the highest bid
        .exec();

        const currentMaxBidAmount = maxBid.length > 0 ? maxBid[0].bidAmount : startPrice;


        const setting = await this.settingModel.findOne({
          fromamount: { $lte: currentMaxBidAmount },
          toamount: { $gte: currentMaxBidAmount },
        });

        const bidIncrement = setting ? setting.bidincrement : 0;

        const nextBidAmount = currentMaxBidAmount + bidIncrement;


        if(usermaxbidamount<nextBidAmount) {
          return {
            statuscode:'failure',
            message: 'Your maximum bid limit is less than next bid amount',
          };
        } else {
          if (product.lotstart == '1' && product.extendAuctionOnBid) {
            const extendAuctionForSeconds = product.extendAuctionForSeconds || 20;

            // Find the last bid for the product
            const lastBid = await this.bidModel.find({ product: productId })
            .sort({ createdAt: -1 }) // Sort by createdAt in descending order
            .limit(1)
            .exec();

              console.log(lastBid);

            if (lastBid.length > 0) {
              const lastBidTime = new Date(lastBid[0].createdAt);
              const timeDifference = (currentTime.getTime() - lastBidTime.getTime()) / 1000; // Difference in seconds
              console.log(timeDifference);
              console.log(extendAuctionForSeconds);
              if (timeDifference > extendAuctionForSeconds) {
                if(lastBid[0].bidAmount<product.reservePrice && product.reservePrice>product.startPrice) {
                  const newBid = new this.bidModel({
                    user: user,
                    product: productId,
                    bidAmount: nextBidAmount,
                    timestamps: currentTime,
                  });

                  await newBid.save();

                  return {
                    statuscode:'success',
                    message: 'Bid placed successfully. Auction extended as the Reserve Price has not been met',
                    bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                  };
                } else {
                  await this.productModel.findByIdAndUpdate(
                    productId,
                    { finaluser: lastBid[0].user },
                  );
                  await this.productModel.findByIdAndUpdate(
                    productId,
                    { finalamount: lastBid[0].bidAmount },
                  );

                  return {
                    statuscode:'failure',
                    message: `Bidding period has ended for this product due to inactivity for more than ${extendAuctionForSeconds} seconds`,
                  };
                }
              } else {
                const newBid = new this.bidModel({
                  user: user,
                  product: productId,
                  bidAmount: nextBidAmount,
                  timestamps: currentTime,
                });

                await newBid.save();

                return {
                  statuscode:'success',
                  message: 'Bid placed successfully',
                  bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
                };
              }
            } else {
              const newBid = new this.bidModel({
                user: user,
                product: productId,
                bidAmount: nextBidAmount,
                timestamps: currentTime,
              });

              await newBid.save();

              return {
                statuscode:'success',
                message: 'Bid placed successfully',
                bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
              };
            }
          } else {
            const newBid = new this.bidModel({
              user: user,
              product: productId,
              bidAmount: nextBidAmount,
              timestamps: currentTime,
            });

            await newBid.save();

            return {
              statuscode:'success',
              message: 'Bid placed successfully',
              bid: await this.bidModel.find({ product: body.productid }).populate('user').exec(),
            };
          }
          
        }

      } else {
        return {
          statuscode:'failure',
          message: 'Invalid Product Request',
        };
      }
      //return await this.bidModel.find({ product: body.productid }).populate('user').exec();;
    }

  }

  async addBanner(body): Promise<any> {
    await this.productModel.findByIdAndUpdate(
      body.id,
      { isbanner: 'yes' },
    );
    return {
      statuscode:'success',
      message: 'Product will be available in Home Banner',
    };
  }

  async removeBanner(body): Promise<any> {
    await this.productModel.findByIdAndUpdate(
      body.id,
      { isbanner: 'no' },
    );
    return {
      statuscode:'success',
      message: 'Product will be removed from Home Banner',
    };
  }

  async homeBanners(body): Promise<any> {
    return this.productModel.find({isbanner:'yes'}).populate('category').exec();
  }

  async upcomingAuctions(body): Promise<any> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    return this.productModel
    .find({
      $and: [
        { auctionStartDate: { $gt: today } },
        { lotNumber: { $ne: null } }
      ]
    })
    .populate('category')
    .limit(6)
    .exec();
  }

  async liveAuctions(body): Promise<any> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    return this.productModel
    .find({
      $and: [
        { auctionStartDate: { $lt: today } },
        { auctionEndDate: { $gt: today } },
        { lotNumber: { $ne: null } }
      ]
    })
    .populate('category')
    .limit(6)
    .exec();
  }
}